package commands

import (
	"context"
	"fmt"
	"io"
	"math/rand"
	"os"
	"os/signal"
	"strconv"
	"syscall"
	"time"

	"github.com/gofrs/flock"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/app"
	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/config"
	"a.yandex-team.ru/library/go/core/log"
)

var startCmd = &cobra.Command{
	Use:          "start",
	SilenceUsage: true,
	Short:        "Start CAuth-agent",
	RunE: func(cmd *cobra.Command, args []string) error {
		lock, err := aquireLock(cfg.DaemonConfig.PIDFile)
		if err != nil {
			return fmt.Errorf("failed to acquire lock: %w", err)
		}
		defer lock.Unlock()

		// save pid to file for updates triggered by systemd timer
		if err := savePID(os.Getpid(), cfg.DaemonConfig.PIDFile); err != nil {
			return fmt.Errorf("failed to write pid file: %w", err)
		}

		instance, err := app.NewApp(cfg, logger)
		if err != nil {
			return fmt.Errorf("failed to create application: %w", err)
		}

		errChan := make(chan error, 1)
		okChan := make(chan struct{}, 1)
		go func() {
			if err := instance.Start(); err != nil {
				errChan <- fmt.Errorf("failed to start application: %w", err)
			} else {
				okChan <- struct{}{}
			}
		}()

		sigChan := make(chan os.Signal, 1)
		signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM, syscall.SIGUSR1)

		defer logger.Info("CAuthURL-agent stopped")

		rand.Seed(time.Now().UnixNano())
		nextDelay := getNextDelay(cfg)
		updateTimer := time.NewTimer(nextDelay)
		yasmTicker := time.NewTicker(cfg.YASMConfig.PushInterval)
		if !cfg.YASMConfig.PushEnabled {
			yasmTicker.Stop()
		}
		for {
			select {
			case s := <-sigChan:
				switch s {
				case syscall.SIGUSR1:
					logger.Info("Reloading via signal")
					if err := instance.Reload(); err != nil {
						logger.Error("Reload failed", log.Error(err))
					} else {
						logger.Info("Successfully reloaded via signal")
					}
				default:
					// TODO: shut down instantly to get faster restart/socket activation?
					logger.Info("Shutting down gracefully by signal")
					return func() error {
						// use context with timeout trying to shut down gracefully
						ctx, cancel := context.WithTimeout(context.Background(), config.ShutdownDeadline)
						defer cancel()
						return instance.Shutdown(ctx)
					}()
				}
			case <-okChan:
				return nil
			case err := <-errChan:
				return err
			case <-updateTimer.C:
				{
					logger.Info("starting data update via timer")
					nextDelay = getNextDelay(cfg)
					if err := instance.Reload(); err != nil {
						logger.Errorf("failed to update data: %s, next update in %s", err, nextDelay.String())
					} else {
						logger.Infof("data was successfully updated, next update in %s", nextDelay.String())
					}
					updateTimer.Reset(nextDelay)
				}
			case <-yasmTicker.C:
				ctx, cancel := context.WithTimeout(context.Background(), time.Second)
				err = instance.PushMetrics(ctx)
				if err != nil {
					logger.Warnf("Failed to push yasm metrics: %s", err)
				}
				cancel()
			}
		}
	},
}

func init() {
	rootCmd.AddCommand(startCmd)
}

func aquireLock(PIDFile string) (*flock.Flock, error) {
	lock := flock.New(PIDFile)
	if locked, err := lock.TryLock(); !locked {
		if err != nil {
			return nil, err
		}
		runningPid := "unknown pid"
		f, err := os.Open(PIDFile)
		if err == nil {
			defer f.Close()
			buf, err := io.ReadAll(f)
			if err == nil {
				runningPid = string(buf)
			}
		}
		return nil, fmt.Errorf("agent is already running with pid: %s", runningPid)
	}
	return lock, nil
}

func savePID(pid int, PIDFile string) error {
	file, err := os.OpenFile(PIDFile, os.O_CREATE|os.O_RDWR|os.O_TRUNC, 0o644)
	if err != nil {
		return err
	}

	defer file.Close()

	_, err = file.WriteString(strconv.Itoa(pid))
	if err != nil {
		return err
	}

	err = file.Sync()
	if err != nil {
		return err
	}
	return file.Chmod(0o644)
}

func getNextDelay(cfg *config.Config) time.Duration {
	return time.Duration(cfg.DaemonConfig.UpdateIntervalSec+rand.Intn(cfg.DaemonConfig.UpdateRandomizedDelaySec)) * time.Second
}
