package config

import (
	"bufio"
	"context"
	"fmt"
	"io"
	"os"
	"strings"
	"time"

	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/file"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
)

const (
	ShutdownDeadline      = 1 * time.Minute
	LegacySourcesConfPath = "/etc/cauth/cauth.conf"
)

type LogConfig struct {
	Level  string `yaml:"level"`
	Output string `yaml:"output"`
}

type RepoConfig struct {
	KeepVersions int    `yaml:"keepVersions"`
	Path         string `yaml:"path"`
}

type CAuthConfig struct {
	URL           string `yaml:"url"`
	LegacySources string `yaml:"legacySources"`
}

type DaemonConfig struct {
	SocketPath                string `yaml:"socketPath"`
	PIDFile                   string `yaml:"pidFile"`
	EnableSocketActivation    bool   `yaml:"enableSocketActivation"`
	UpdateIntervalSec         int    `yaml:"updateIntervalSec"`
	UpdateRandomizedDelaySec  int    `yaml:"updateRandomizedDelaySec"`
	MaxUpdatesPerMinute       int    `yaml:"maxUpdatesPerMinute"`
	UpdateHistoryFile         string `yaml:"updateTrackingFile"`
	ResolveGroupMembers       bool   `yaml:"resolveGroupMembers"`
	VirtualUsersShellOverride string `yaml:"virtualUsersShellOverride"`
	VirtualUsersGIDOverride   int    `yaml:"virtualUsersGidOverride"`
	RealUsersGIDOverride      int    `yaml:"realUsersGidOverride"`
	RealUsersShellOverride    string `yaml:"realUsersShellOverride"`
}

type YASMConfig struct {
	PushEnabled  bool          `yaml:"pushEnabled"`
	CType        string        `yaml:"ctype"`
	IType        string        `yaml:"itype"`
	YASMURL      string        `yaml:"yasmURL"`
	PushInterval time.Duration `yaml:"pushInterval"`
}

type Config struct {
	Log          LogConfig    `yaml:"log"`
	RepoConfig   RepoConfig   `yaml:"repoConfig"`
	CAuthConfig  CAuthConfig  `yaml:"cAuthConfig"`
	DaemonConfig DaemonConfig `yaml:"daemonConfig"`
	YASMConfig   YASMConfig   `yaml:"YASMConfig"`
}

// getLegacySources parses legacy cauth.conf to extract value of sources variable.
// cauth.conf could have multiple sources variables, but we are interested in the last ones.
func getLegacySources(reader io.Reader) string {
	var rv string
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		if scanner.Err() != nil {
			return rv
		}
		line := scanner.Text()
		if strings.HasPrefix(line, "sources=") {
			parts := strings.Split(line, "=")
			if len(parts) == 2 {
				rv = strings.Trim(parts[1], `"'`)
			}
		}
	}
	return rv
}

func DefaultConfig() *Config {
	return &Config{
		Log: LogConfig{
			Level:  log.InfoString,
			Output: "stdout",
		},
		RepoConfig: RepoConfig{
			KeepVersions: 5,
			Path:         "/var/cache/yandex-cauth-userd",
		},
		CAuthConfig: CAuthConfig{
			URL: "https://cauth.yandex.net:4443",
		},
		DaemonConfig: DaemonConfig{
			SocketPath:                "/run/yandex-cauth-userd.sock",
			PIDFile:                   "/run/yandex-cauth-userd.pid",
			EnableSocketActivation:    true,
			UpdateIntervalSec:         600,
			UpdateRandomizedDelaySec:  600,
			MaxUpdatesPerMinute:       3,
			UpdateHistoryFile:         "/run/yandex-cauth-userd-updates.json",
			ResolveGroupMembers:       true,
			VirtualUsersShellOverride: "",
			VirtualUsersGIDOverride:   -1,
			RealUsersGIDOverride:      -1,
			RealUsersShellOverride:    "",
		},
		YASMConfig: YASMConfig{
			PushEnabled:  false,
			CType:        "unknown",
			IType:        "unknown",
			YASMURL:      "http://localhost:11005",
			PushInterval: 5 * time.Second,
		},
	}
}

func NewConfig(configPath string, legacyConfPath string) (*Config, error) {
	cfg := DefaultConfig()
	if f, err := os.Open(legacyConfPath); err == nil {
		defer f.Close()
		cfg.CAuthConfig.LegacySources = getLegacySources(f)
	}
	if configPath == "" {
		return cfg, nil
	}

	loader := confita.NewLoader(file.NewBackend(configPath))
	if err := loader.Load(context.Background(), cfg); err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}

	return cfg, nil
}

func (l *LogConfig) ZapConfig() zap.Config {
	var lvl zapcore.Level
	switch strings.ToLower(l.Level) {
	case log.TraceString:
		lvl = zapcore.DebugLevel
	case log.DebugString:
		lvl = zapcore.DebugLevel
	case log.InfoString:
		lvl = zapcore.InfoLevel
	case log.WarnString:
		lvl = zapcore.WarnLevel
	case log.ErrorString:
		lvl = zapcore.ErrorLevel
	case log.FatalString:
		lvl = zapcore.FatalLevel
	default:
		lvl = zapcore.InfoLevel
	}

	return zap.Config{
		Level:            zap.NewAtomicLevelAt(lvl),
		Encoding:         "kv",
		OutputPaths:      []string{l.Output},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			NameKey:        "logger",
			LineEnding:     zapcore.DefaultLineEnding,
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.SecondsDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	}
}
