package config

import (
	"io/ioutil"
	"os"
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestGetLegacySources(t *testing.T) {
	r := strings.NewReader(`sources="cms,idm,walle"`)
	assert.Equal(t, "cms,idm,walle", getLegacySources(r))

	r = strings.NewReader(`sources='cms,idm,walle'`)
	assert.Equal(t, "cms,idm,walle", getLegacySources(r))

	r = strings.NewReader(`# This file is managed by Salt.

# https://st.yandex-team.ru/RUNTIMECLOUD-4840
# https://st.yandex-team.ru/WALLE-2793
sources="idm-cms,walle"`)

	assert.Equal(t, "idm-cms,walle", getLegacySources(r))
}

func TestNewConfig_LegacyExists(t *testing.T) {
	legacyFile, err := ioutil.TempFile(".", "*legacy.conf")
	assert.NoError(t, err)
	defer legacyFile.Close()
	defer os.Remove(legacyFile.Name())
	confFile, err := ioutil.TempFile(".", "*cauth.yaml")
	assert.NoError(t, err)
	defer confFile.Close()
	defer os.Remove(confFile.Name())
	_, _ = legacyFile.WriteString(`sources="idm-cms,walle"`)
	_, _ = confFile.WriteString(`{}`)
	c, err := NewConfig(confFile.Name(), legacyFile.Name())
	assert.NoError(t, err)
	assert.Equal(t, "idm-cms,walle", c.CAuthConfig.LegacySources)
}

func TestNewConfig_LegacyDoesNotExist(t *testing.T) {
	confFile, err := ioutil.TempFile(".", "*cauth.yaml")
	assert.NoError(t, err)
	defer confFile.Close()
	defer os.Remove(confFile.Name())
	_, _ = confFile.WriteString(`{}`)
	c, err := NewConfig(confFile.Name(), LegacySourcesConfPath)
	assert.NoError(t, err)
	assert.Equal(t, "", c.CAuthConfig.LegacySources)
}
