package storage

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"os"
	"time"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/cauth"
)

// Controller maintains Index instance.
type Controller interface {
	// HasIndex checks if provided Repo has current Version available.
	HasIndex(repo Repo) bool
	// LoadIndex loads Index from provided Repo to Controller.
	LoadIndex(ctx context.Context, repo Repo) error
	// UpdateIndex fetches new Version from provided cauth.Client, stores it to Repo,
	// updates current Version in Repo and updates Index stored in Controller
	// from new current Version
	UpdateIndex(ctx context.Context, client cauth.Client, repo Repo) error
	// GetIndex returns data stored in Controller
	GetIndex() (Index, error)
}

type updateHistory struct {
	Updates []time.Time `json:"updates"`
}

// controller provides simple implementation of Controller interface.
type controller struct {
	idx                 Index
	maxUpdatesPerMinute int
	updateHistoryFile   string
	resolveGroupMembers bool
	overrides           *UserOverrides
}

// UserOverrides contains overrides of specific user fields
type UserOverrides struct {
	VirtualUsersShellOverride string
	VirtualUsersGIDOverride   int
	RealUsersGIDOverride      int
	RealUsersShellOverride    string
}

// emptyOverrides return UserOverrides structure as if overrides absent in config.
func emptyOverrides() *UserOverrides {
	return &UserOverrides{RealUsersGIDOverride: -1, VirtualUsersGIDOverride: -1}
}

func NewController(maxUpdatesPerMinute int, updateHistoryFile string, resolveGroupMembers bool, overrides *UserOverrides) *controller {
	return &controller{maxUpdatesPerMinute: maxUpdatesPerMinute,
		updateHistoryFile:   updateHistoryFile,
		resolveGroupMembers: resolveGroupMembers,
		overrides:           overrides,
	}
}

func (c *controller) HasIndex(repo Repo) bool {
	_, err := repo.CurrentVersion()
	return err == nil
}

func (c *controller) LoadIndex(ctx context.Context, repo Repo) error {
	v, err := repo.CurrentVersion()
	if err != nil {
		return err
	}
	idx, err := NewInMemoryIndex(ctx, v, c.resolveGroupMembers, c.overrides)
	if err != nil {
		return err
	}
	c.idx = idx
	return nil
}

func (c *controller) UpdateIndex(ctx context.Context, client cauth.Client, repo Repo) error {
	h, err := c.loadUpdateHistory()
	if err != nil {
		return err
	}
	if updateCount := h.updatesLastMinute(); updateCount < c.maxUpdatesPerMinute {
		h.addUpdate(time.Now())
		err = c.saveUpdateHistory(h)
		if err != nil {
			return err
		}
		newVer := NewRemoteVersion(client)
		err := repo.PersistVersion(ctx, newVer)
		if err != nil {
			return err
		}
		err = repo.UpdateCurrent(newVer)
		if err != nil {
			return err
		}
		return c.LoadIndex(ctx, repo)
	} else {
		return fmt.Errorf("failed to update index, updates last minute: %d, max updates last minute: %d", updateCount, c.maxUpdatesPerMinute)
	}
}

func (c *controller) GetIndex() (Index, error) {
	if c.idx == nil {
		return nil, fmt.Errorf("controller is not initialized with any data")
	}
	return c.idx, nil
}

func (c *controller) loadUpdateHistory() (*updateHistory, error) {
	f, err := os.Open(c.updateHistoryFile)
	if err != nil && errors.Is(err, os.ErrNotExist) {
		return &updateHistory{}, nil
	}
	if err != nil {
		return nil, err
	}
	defer f.Close()
	if buf, err := io.ReadAll(f); err == nil {
		rv := &updateHistory{}
		err = json.Unmarshal(buf, rv)
		if err != nil {
			return nil, err
		}
		return rv, nil
	} else {
		return nil, err
	}
}

func (c *controller) saveUpdateHistory(h *updateHistory) error {
	f, err := os.OpenFile(c.updateHistoryFile, os.O_CREATE|os.O_TRUNC|os.O_RDWR, 0o644)
	if err != nil {
		return err
	}
	defer f.Close()
	// Shrink history before saving
	keepHistoryRecords := 2 * c.maxUpdatesPerMinute
	historyRecords := len(h.Updates)
	if historyRecords > keepHistoryRecords {
		h.Updates = h.Updates[historyRecords-keepHistoryRecords:]
	}
	if buf, err := json.Marshal(h); err == nil {
		_, err = f.Write(buf)
		if err != nil {
			return err
		}
		err = f.Sync()
		if err != nil {
			return err
		}
		return nil
	} else {
		return err
	}
}

func (h *updateHistory) updatesLastMinute() int {
	t := time.Now().Add(-time.Minute)
	updateCount := 0
	for _, u := range h.Updates {
		if u.After(t) {
			updateCount++
		}
	}
	return updateCount
}

func (h *updateHistory) addUpdate(t time.Time) {
	h.Updates = append(h.Updates, t)
}
