package storage

import (
	"context"
	"fmt"
	"os"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

const historyFilePath = "yandex-cauth-userd-updates.json"

type controllerRepoMock struct {
	cv       Version
	cvErr    error
	cvCalled bool
	pvErr    error
	pvCalled bool
	ucErr    error
	ucCalled bool
}

func (r *controllerRepoMock) CurrentVersion() (Version, error) {
	r.cvCalled = true
	return r.cv, r.cvErr
}

func (r *controllerRepoMock) PersistVersion(ctx context.Context, version Version) error {
	r.pvCalled = true
	return r.pvErr
}

func (r *controllerRepoMock) UpdateCurrent(version Version) error {
	r.ucCalled = true
	return r.ucErr
}

func TestController_HasData(t *testing.T) {
	repo := &controllerRepoMock{}
	c := NewController(3, historyFilePath, true, emptyOverrides())
	assert.True(t, c.HasIndex(repo))
	repo.cvErr = fmt.Errorf("mock")
	assert.False(t, c.HasIndex(repo))
}

func TestController_LoadData(t *testing.T) {
	repo := &controllerRepoMock{}
	ts, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	repo.cv = &vMock{ts: ts}
	c := NewController(3, historyFilePath, true, emptyOverrides())
	_, err = c.GetIndex()
	assert.Error(t, err)
	err = c.LoadIndex(context.TODO(), repo)
	assert.NoError(t, err)
	d, err := c.GetIndex()
	assert.NoError(t, err)
	assert.NotNil(t, d)
}

func TestController_UpdateData(t *testing.T) {
	repo := &controllerRepoMock{}
	ts, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	repo.cv = &vMock{ts: ts}
	c := NewController(3, historyFilePath, true, emptyOverrides())
	_, err = c.GetIndex()
	assert.Error(t, err)
	err = c.UpdateIndex(context.TODO(), nil, repo)
	assert.NoError(t, err)
	assert.True(t, repo.cvCalled)
	assert.True(t, repo.ucCalled)
	assert.True(t, repo.pvCalled)
	d, err := c.GetIndex()
	assert.NoError(t, err)
	assert.NotNil(t, d)
	_ = os.Remove(historyFilePath)
}

func TestUpdateHistory(t *testing.T) {
	now := time.Now()
	h := &updateHistory{Updates: []time.Time{
		now.Add(-time.Minute * 15),
		now.Add(-time.Minute * 10),
		now.Add(-time.Second * 15),
	}}
	assert.Equal(t, 1, h.updatesLastMinute())
	h.addUpdate(now)
	assert.Equal(t, 2, h.updatesLastMinute())
}

func TestController_UpdateData_History(t *testing.T) {
	repo := &controllerRepoMock{}
	ts, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	repo.cv = &vMock{ts: ts}
	c := NewController(3, historyFilePath, true, emptyOverrides())
	_, err = c.GetIndex()
	assert.Error(t, err)
	err = c.UpdateIndex(context.TODO(), nil, repo)
	assert.NoError(t, err)

	h, err := c.loadUpdateHistory()
	assert.NoError(t, err)
	assert.Equal(t, 1, len(h.Updates))
	err = c.UpdateIndex(context.TODO(), nil, repo)
	assert.NoError(t, err)
	h, err = c.loadUpdateHistory()
	assert.NoError(t, err)
	assert.Equal(t, 2, len(h.Updates))
	_ = os.Remove(historyFilePath)
}

func TestController_UpdateData_HistoryShrink(t *testing.T) {
	repo := &controllerRepoMock{}
	ts, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	repo.cv = &vMock{ts: ts}
	c := NewController(3, historyFilePath, true, emptyOverrides())
	_, err = c.GetIndex()
	assert.Error(t, err)
	err = c.UpdateIndex(context.TODO(), nil, repo)
	assert.NoError(t, err)

	h, err := c.loadUpdateHistory()
	assert.NoError(t, err)
	assert.Equal(t, 1, len(h.Updates))
	now := time.Now().Add(-time.Hour)
	h.Updates = append([]time.Time{now, now, now, now, now, now}, h.Updates...)
	err = c.saveUpdateHistory(h)
	assert.NoError(t, err)
	h, err = c.loadUpdateHistory()
	assert.NoError(t, err)
	assert.Equal(t, 6, len(h.Updates))
	_ = os.Remove(historyFilePath)
}
