package storage

import (
	"context"
	"io"
	"strings"
	"testing"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/pkg/cauthrpc"
	"github.com/golang/protobuf/proto"
	"github.com/stretchr/testify/assert"
)

//N(o)S(taff) I(nsecure) S(ecure)
type keysInfoTestMockNSIS struct {
}

func (d *keysInfoTestMockNSIS) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:Anton Suvorov:/home/warwish:/bin/bash")}, nil
}

func (d *keysInfoTestMockNSIS) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("svc_dist:x:58760:nekto0n,noiseless,sivanichkin,vaspahomov,warwish")}, nil
}

func (d *keysInfoTestMockNSIS) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchAccess should be unused")
}

func (d *keysInfoTestMockNSIS) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish : ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97")}, nil
}

func (d *keysInfoTestMockNSIS) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:Anton Suvorov:/home/warwish:/bin/bash")}, nil
}

func (d *keysInfoTestMockNSIS) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("%dpt_yandex_mnt_sa_runtime_mondev_6921 ALL=(ALL) ALL")}, nil
}

func (d *keysInfoTestMockNSIS) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("{\n  \"insecure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure\",\n  \"sudo_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo\",\n  \"krl_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst\",\n  \"key_sources\": [\n    \"insecure\",\n    \"secure\"\n  ],\n  \"secure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure\"\n}")}, nil
}

func (d *keysInfoTestMockNSIS) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBP8w+sr7XJuSXHQ9OAOj0eRfv4fQi/qFnW185Ae5fkKX9VhtmkM7LpfIy7NeOOthxS9wUJmfEcAGUrSH5Pry/ZU=\necdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBEJARUiEOnhTZ1d9A9fPIAN7NNOOUantJ5jvr0zjRCc2Qf7RFJUOvT9NaTUIilhkfDviLJj4mp07QWK23pz8OXE= insecure_20211011")}, nil
}

func (d *keysInfoTestMockNSIS) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBJnF1UG50PejZLaFFAWTMOL7e4xy44Z/mDJyF6RTsQsIxFN2oC9E4cwOTKSf2Ko/jemdnDOWu+j8X6f4y2KTV5M=\necdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBEhN1c1fB+DiAH/hFpOV4oUSyBUZ/KkIa1VpHOzRx54YBQMi+7cMBFdqYMSQO4zQAsQk1uix76/XLz90G1WEGuI= secure_20211011")}, nil
}

func (d *keysInfoTestMockNSIS) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchSudo should be unused")
}

func (d *keysInfoTestMockNSIS) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchKRL should be unused")
}

//S(taff) I(nsecure) N(o)S(ecure)
type keysInfoTestMockSINS struct {
}

func (d *keysInfoTestMockSINS) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:Anton Suvorov:/home/warwish:/bin/bash")}, nil
}

func (d *keysInfoTestMockSINS) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("svc_dist:x:58760:nekto0n,noiseless,sivanichkin,vaspahomov,warwish")}, nil
}

func (d *keysInfoTestMockSINS) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchAccess should be unused")
}

func (d *keysInfoTestMockSINS) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish : ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97")}, nil
}

func (d *keysInfoTestMockSINS) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:Anton Suvorov:/home/warwish:/bin/bash")}, nil
}

func (d *keysInfoTestMockSINS) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("%dpt_yandex_mnt_sa_runtime_mondev_6921 ALL=(ALL) ALL")}, nil
}

func (d *keysInfoTestMockSINS) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("{\n  \"insecure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure\",\n  \"sudo_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo\",\n  \"krl_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst\",\n  \"key_sources\": [\n    \"staff\",\n    \"insecure\"\n  ],\n  \"secure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure\"\n}")}, nil
}

func (d *keysInfoTestMockSINS) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBP8w+sr7XJuSXHQ9OAOj0eRfv4fQi/qFnW185Ae5fkKX9VhtmkM7LpfIy7NeOOthxS9wUJmfEcAGUrSH5Pry/ZU=\necdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBEJARUiEOnhTZ1d9A9fPIAN7NNOOUantJ5jvr0zjRCc2Qf7RFJUOvT9NaTUIilhkfDviLJj4mp07QWK23pz8OXE= insecure_20211011")}, nil
}

func (d *keysInfoTestMockSINS) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchSecure should be unused")
}

func (d *keysInfoTestMockSINS) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchSudo should be unused")
}

func (d *keysInfoTestMockSINS) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchKRL should be unused")
}

type dataTestMock struct {
}

func (d *dataTestMock) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:Anton Suvorov:/home/warwish:/bin/bash")}, nil
}

func (d *dataTestMock) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("svc_dist:x:58760:nekto0n,noiseless,sivanichkin,vaspahomov,warwish")}, nil
}

func (d *dataTestMock) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchAccess should be unused")
}

func (d *dataTestMock) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish : ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97")}, nil
}

func (d *dataTestMock) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:Anton Suvorov:/home/warwish:/bin/bash")}, nil
}

func (d *dataTestMock) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("%dpt_yandex_mnt_sa_runtime_mondev_6921 ALL=(ALL) ALL")}, nil
}

func (d *dataTestMock) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("{\n  \"insecure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure\",\n  \"sudo_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo\",\n  \"krl_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst\",\n  \"key_sources\": [\n    \"staff\"\n  ],\n  \"secure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure\"\n}")}, nil
}

func (d *dataTestMock) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *dataTestMock) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *dataTestMock) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *dataTestMock) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

type virtualUsersTestMock struct {
}

func (d *virtualUsersTestMock) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("robot-mock:*:30000:30000:robot-mock:/home/robot-mock:/bin/false")}, nil
}

func (d *virtualUsersTestMock) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("dpt_virtual:x:30001:robot-mock")}, nil
}

func (d *virtualUsersTestMock) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchAccess should be unused")
}

func (d *virtualUsersTestMock) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish : ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97")}, nil
}

func (d *virtualUsersTestMock) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:warwish:/home/warwish:/bin/bash")}, nil
}

func (d *virtualUsersTestMock) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("%dpt_yandex_mnt_sa_runtime_mondev_6921 ALL=(ALL) ALL")}, nil
}

func (d *virtualUsersTestMock) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("{\n  \"insecure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure\",\n  \"sudo_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo\",\n  \"krl_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst\",\n  \"key_sources\": [\n    \"staff\",\n    \"insecure\"  ],\n  \"secure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure\"\n}")}, nil
}

func (d *virtualUsersTestMock) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *virtualUsersTestMock) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *virtualUsersTestMock) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *virtualUsersTestMock) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

type realUsersTestMock struct {
}

func (d *realUsersTestMock) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("vpupkin:*:30000:30000:Vasya Pupkin:/home/vpupkin:/bin/dash")}, nil
}

func (d *realUsersTestMock) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("dpt_yandex:x:30001:vpupkin")}, nil
}

func (d *realUsersTestMock) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	panic("FetchAccess should be unused")
}

func (d *realUsersTestMock) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("vpupkin : ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97")}, nil
}

func (d *realUsersTestMock) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("vpupkin:*:53523:239396:Vasya Pupkin:/home/vpupkin:/bin/dash")}, nil
}

func (d *realUsersTestMock) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("%dpt_yandex ALL=(ALL) ALL")}, nil
}

func (d *realUsersTestMock) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("{\n  \"insecure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure\",\n  \"sudo_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo\",\n  \"krl_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst\",\n  \"key_sources\": [\n    \"staff\",\n    \"insecure\"\n  ],\n  \"secure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure\"\n}")}, nil
}

func (d *realUsersTestMock) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *realUsersTestMock) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *realUsersTestMock) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (d *realUsersTestMock) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func TestInMemoryDataWithKIInsecureAndSecure(t *testing.T) {
	client := &keysInfoTestMockNSIS{}
	data, err := NewInMemoryIndex(context.TODO(), client, true, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     53523,
		Gid:     239396,
		Login:   "warwish",
		HomeDir: "/home/warwish",
		Shell:   "/bin/bash",
		Gecos:   "Anton Suvorov",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(53523)
	assert.True(t, ok)
	assert.Equal(t, b, u)

	b, err = proto.Marshal(&cauthrpc.Group{
		Gid:     58760,
		Name:    "svc_dist",
		Members: strings.Split("nekto0n,noiseless,sivanichkin,vaspahomov,warwish", ","),
	})
	assert.NoError(t, err)
	g, ok := data.GroupByName("svc_dist")
	assert.True(t, ok)
	assert.Equal(t, b, g)
	g, ok = data.GroupByGID(58760)
	assert.True(t, ok)
	assert.Equal(t, b, g)

	b, err = proto.Marshal(&cauthrpc.UserGroups{Groups: []uint32{58760}})
	assert.NoError(t, err)
	ug, ok := data.UserGroups("nekto0n")
	assert.True(t, ok)
	assert.Equal(t, b, ug)

	k, ok := data.KeysByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, []byte("cert-authority,principals=\"warwish\" ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBP8w+sr7XJuSXHQ9OAOj0eRfv4fQi/qFnW185Ae5fkKX9VhtmkM7LpfIy7NeOOthxS9wUJmfEcAGUrSH5Pry/ZU=\ncert-authority,principals=\"warwish\" ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBEJARUiEOnhTZ1d9A9fPIAN7NNOOUantJ5jvr0zjRCc2Qf7RFJUOvT9NaTUIilhkfDviLJj4mp07QWK23pz8OXE= insecure_20211011\ncert-authority,principals=\"warwish\" ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBJnF1UG50PejZLaFFAWTMOL7e4xy44Z/mDJyF6RTsQsIxFN2oC9E4cwOTKSf2Ko/jemdnDOWu+j8X6f4y2KTV5M=\ncert-authority,principals=\"warwish\" ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBEhN1c1fB+DiAH/hFpOV4oUSyBUZ/KkIa1VpHOzRx54YBQMi+7cMBFdqYMSQO4zQAsQk1uix76/XLz90G1WEGuI= secure_20211011"), k)
}

func TestInMemoryDataWithKIStaffAndInsecure(t *testing.T) {
	client := &keysInfoTestMockSINS{}
	data, err := NewInMemoryIndex(context.TODO(), client, true, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     53523,
		Gid:     239396,
		Login:   "warwish",
		HomeDir: "/home/warwish",
		Shell:   "/bin/bash",
		Gecos:   "Anton Suvorov",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(53523)
	assert.True(t, ok)
	assert.Equal(t, b, u)

	b, err = proto.Marshal(&cauthrpc.Group{
		Gid:     58760,
		Name:    "svc_dist",
		Members: strings.Split("nekto0n,noiseless,sivanichkin,vaspahomov,warwish", ","),
	})
	assert.NoError(t, err)
	g, ok := data.GroupByName("svc_dist")
	assert.True(t, ok)
	assert.Equal(t, b, g)
	g, ok = data.GroupByGID(58760)
	assert.True(t, ok)
	assert.Equal(t, b, g)

	b, err = proto.Marshal(&cauthrpc.UserGroups{Groups: []uint32{58760}})
	assert.NoError(t, err)
	ug, ok := data.UserGroups("nekto0n")
	assert.True(t, ok)
	assert.Equal(t, b, ug)

	k, ok := data.KeysByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, []byte("ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97\ncert-authority,principals=\"warwish\" ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBP8w+sr7XJuSXHQ9OAOj0eRfv4fQi/qFnW185Ae5fkKX9VhtmkM7LpfIy7NeOOthxS9wUJmfEcAGUrSH5Pry/ZU=\ncert-authority,principals=\"warwish\" ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBEJARUiEOnhTZ1d9A9fPIAN7NNOOUantJ5jvr0zjRCc2Qf7RFJUOvT9NaTUIilhkfDviLJj4mp07QWK23pz8OXE= insecure_20211011"), k)
}

func TestInMemoryData(t *testing.T) {
	client := &dataTestMock{}
	data, err := NewInMemoryIndex(context.TODO(), client, true, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     53523,
		Gid:     239396,
		Login:   "warwish",
		HomeDir: "/home/warwish",
		Shell:   "/bin/bash",
		Gecos:   "Anton Suvorov",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(53523)
	assert.True(t, ok)
	assert.Equal(t, b, u)

	b, err = proto.Marshal(&cauthrpc.Group{
		Gid:     58760,
		Name:    "svc_dist",
		Members: strings.Split("nekto0n,noiseless,sivanichkin,vaspahomov,warwish", ","),
	})
	assert.NoError(t, err)
	g, ok := data.GroupByName("svc_dist")
	assert.True(t, ok)
	assert.Equal(t, b, g)
	g, ok = data.GroupByGID(58760)
	assert.True(t, ok)
	assert.Equal(t, b, g)

	b, err = proto.Marshal(&cauthrpc.UserGroups{Groups: []uint32{58760}})
	assert.NoError(t, err)
	ug, ok := data.UserGroups("nekto0n")
	assert.True(t, ok)
	assert.Equal(t, b, ug)

	k, ok := data.KeysByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, []byte("ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97"), k)
}

func TestInMemoryData_ResolveGroupMembersFalse(t *testing.T) {
	client := &dataTestMock{}
	data, err := NewInMemoryIndex(context.TODO(), client, false, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     53523,
		Gid:     239396,
		Login:   "warwish",
		HomeDir: "/home/warwish",
		Shell:   "/bin/bash",
		Gecos:   "Anton Suvorov",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(53523)
	assert.True(t, ok)
	assert.Equal(t, b, u)

	b, err = proto.Marshal(&cauthrpc.Group{
		Gid:  58760,
		Name: "svc_dist",
	})
	assert.NoError(t, err)
	g, ok := data.GroupByName("svc_dist")
	assert.True(t, ok)
	assert.Equal(t, b, g)
	g, ok = data.GroupByGID(58760)
	assert.True(t, ok)
	assert.Equal(t, b, g)

	b, err = proto.Marshal(&cauthrpc.UserGroups{Groups: []uint32{58760}})
	assert.NoError(t, err)
	ug, ok := data.UserGroups("nekto0n")
	assert.True(t, ok)
	assert.Equal(t, b, ug)

	k, ok := data.KeysByLogin("warwish")
	assert.True(t, ok)
	assert.Equal(t, []byte("ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAQC/jEZapPBPqmANGZvhwheVq1R1vZEccME0sqyeGVODJz3F4s059L0OLWA8kZhf/6I8OPtbMa6y0YMdhOVRwV1q33Pw6qcEBY7KcOBgtXNCO4QHAlH14FIVomi+PC4QSgbNJVbfL+5iO5EwhKaQ5R09uwkk0Z1sg1ZoPffY7S1GKgFyOrIApIcBKqL7X5FZJyGICueWzhebXAUME1hY5iBUthIdkAr/vReXrnPIMq/8XAr+dfgJTYj7lXKxNBLMDHyBwQE07RtLdwcGmGSh3AdRbErDZF7k5xFDzFQm0ewHb67uN0zSGAPinZY/G0GbkfaTLFCmRDZsr0FlT0yCresPd8c7wGyzth/AyWkYcRzTNDboieZ7kPQJMtYGISNyQi8Dgh81mdq8MRkpYa1WnAnHRBk2zq4QaYMGJDy1S58xbc5vozK4t8MeXPANonyHyACSDHl5yU0Yp0piYJ4PK9HRoYsIvAj2Tdymrk3ucHmristeMAvvD7yIzbaPyj2THg0AUCF7bJZ6ovqpzSyJfCdwPckQxenw/tK1Yk5ehJ9iDRrSR3Lc+4NBWNqw49vxEgd8m+kjNSjU+x9ff+qUzPxE5QLKNQnl28JaPtqm2iAQ5Xglsrwx6PeQRoLxBb1jwNcrtMfBi5zTFYwxZ482tua+1s29RMarFfNzgyZyMbcjMY6ie4KwmJYZ+azJVabxLr1nLX8Jw0S1dTxwcjANOiOOq+TzUml+amUDrDiZyXqQF+seBx+fBG46y4hcMgdgEnkVGrDG9Ku+rDU2Iw2qcBuPEA9Db2aIeK6YlmE2PgpnXiak6lEfti6+gR9aDwG98CQE0FgrVdBYzPb7wTjFAQr22+whAmARREhiz3kqU/oPYcqrkrImdoF3HQtWcQfWAxvgowIz4M+sTy+7m6idDASsdXcyIR1rM81P83uGgsMWm177tpH5Kg0+L5fFKJ/Qlw5A5HDOX7g/qrlvvB0KbMZzn2XLkMsF+aNlNL+lx80tDRQOleJCjNQ959QT7P86Qjs09xQjcDmURf93gBm1xJZZDrB3aOEbzVl8aBtbb+7I/FOOxJqKpXS0nJmC9lHDZ2T4Kkp6yJubovSWffDFq8WxPwSN5nIRaV74m45DDZySe6kFU/ZdJUzjebWpA0Kp0OqqNjw9ezk6ygL8iI9C6JygzAKwLBk0l2qOpdo2W9W4GBjx/WNAtIa6eJPgQSesRT7MRtx2VdPQ6AQ2LM8p071s/TjGZOozQZoZq/km9gy6M+UW3sUFcDnJBIjkgDtxmAtHxuAZrJijh2tYhke6ufamxx56ckO5xyNlEhr+vluCEqFTkpJipN4nLkMzO7/BBr9jFnmywT/yUJqYJv2OhK97"), k)
}

func TestInMemoryData_ServeradminsGroup(t *testing.T) {
	client := &dataTestMock{}
	data, err := NewInMemoryIndex(context.TODO(), client, true, emptyOverrides())
	assert.NoError(t, err)
	b, err := proto.Marshal(&cauthrpc.Group{
		Gid:     serveradminsGID,
		Name:    "serveradmins",
		Members: []string{"warwish"},
	})
	assert.NoError(t, err)
	g, ok := data.GroupByName("serveradmins")
	assert.True(t, ok)
	assert.Equal(t, b, g)
	g, ok = data.GroupByGID(serveradminsGID)
	assert.True(t, ok)
	assert.Equal(t, b, g)
}

func TestIsVirtualUser(t *testing.T) {
	assert.True(t, isVirtualUser("robot-mock", nil))
	assert.True(t, isVirtualUser("zomb-mock", nil))
	assert.True(t, isVirtualUser("mockrobot", map[string]struct{}{"mockrobot": {}}))
}

func TestInMemoryData_VirtualUsersShellOverridePresent(t *testing.T) {
	client := &virtualUsersTestMock{}
	overrides := emptyOverrides()
	overrides.VirtualUsersShellOverride = "/bin/bash"
	data, err := NewInMemoryIndex(context.TODO(), client, false, overrides)
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     30000,
		Login:   "robot-mock",
		HomeDir: "/home/robot-mock",
		Shell:   "/bin/bash",
		Gecos:   "robot-mock",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("robot-mock")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func TestInMemoryData_VirtualUsersShellOverrideAbsent(t *testing.T) {
	client := &virtualUsersTestMock{}
	data, err := NewInMemoryIndex(context.TODO(), client, false, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     30000,
		Login:   "robot-mock",
		HomeDir: "/home/robot-mock",
		Shell:   "/bin/false",
		Gecos:   "robot-mock",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("robot-mock")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func TestInMemoryData_RealUsersShellOverridePresent(t *testing.T) {
	client := &realUsersTestMock{}
	overrides := emptyOverrides()
	overrides.RealUsersShellOverride = "/bin/bash"
	data, err := NewInMemoryIndex(context.TODO(), client, false, overrides)
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     30000,
		Login:   "vpupkin",
		HomeDir: "/home/vpupkin",
		Shell:   "/bin/bash",
		Gecos:   "Vasya Pupkin",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("vpupkin")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func TestInMemoryData_RealUsersShellOverrideAbsent(t *testing.T) {
	client := &realUsersTestMock{}
	data, err := NewInMemoryIndex(context.TODO(), client, false, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     30000,
		Login:   "vpupkin",
		HomeDir: "/home/vpupkin",
		Shell:   "/bin/dash",
		Gecos:   "Vasya Pupkin",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("vpupkin")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func TestInMemoryData_RealUsersGIDOverridePresent(t *testing.T) {
	client := &realUsersTestMock{}
	overrides := emptyOverrides()
	overrides.RealUsersGIDOverride = 0
	data, err := NewInMemoryIndex(context.TODO(), client, false, overrides)
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     0,
		Login:   "vpupkin",
		HomeDir: "/home/vpupkin",
		Shell:   "/bin/dash",
		Gecos:   "Vasya Pupkin",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("vpupkin")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func TestInMemoryData_RealUsersGIDOverrideAbsent(t *testing.T) {
	client := &realUsersTestMock{}
	data, err := NewInMemoryIndex(context.TODO(), client, false, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     30000,
		Login:   "vpupkin",
		HomeDir: "/home/vpupkin",
		Shell:   "/bin/dash",
		Gecos:   "Vasya Pupkin",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("vpupkin")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func TestInMemoryData_VirtualUsersGIDOverridePresent(t *testing.T) {
	client := &virtualUsersTestMock{}
	overrides := emptyOverrides()
	overrides.VirtualUsersGIDOverride = 0
	data, err := NewInMemoryIndex(context.TODO(), client, false, overrides)
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     0,
		Login:   "robot-mock",
		HomeDir: "/home/robot-mock",
		Shell:   "/bin/false",
		Gecos:   "robot-mock",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("robot-mock")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func TestInMemoryData_VirtualUsersGIDOverrideAbsent(t *testing.T) {
	client := &virtualUsersTestMock{}
	data, err := NewInMemoryIndex(context.TODO(), client, false, emptyOverrides())
	assert.NoError(t, err)

	b, err := proto.Marshal(&cauthrpc.User{
		Uid:     30000,
		Gid:     30000,
		Login:   "robot-mock",
		HomeDir: "/home/robot-mock",
		Shell:   "/bin/false",
		Gecos:   "robot-mock",
	})
	assert.NoError(t, err)
	u, ok := data.UserByLogin("robot-mock")
	assert.True(t, ok)
	assert.Equal(t, b, u)
	u, ok = data.UserByUID(30000)
	assert.True(t, ok)
	assert.Equal(t, b, u)
}

func BenchmarkUnserializedStorage(b *testing.B) {
	group := &cauthrpc.Group{
		Gid:     58760,
		Name:    "svc_dist",
		Members: strings.Split("nekto0n,noiseless,sivanichkin,vaspahomov,warwish", ","),
	}
	groupsByName := map[string]*cauthrpc.Group{"svc_dist": group}
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		if g, ok := groupsByName["svc_dist"]; ok {
			_, _ = proto.Marshal(g)
		}
	}
}

func BenchmarkSerializedStorage(b *testing.B) {
	group := &cauthrpc.Group{
		Gid:     58760,
		Name:    "svc_dist",
		Members: strings.Split("nekto0n,noiseless,sivanichkin,vaspahomov,warwish", ","),
	}
	buf, _ := proto.Marshal(group)
	groupsByName := map[string][]byte{"svc_dist": buf}
	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		if g, ok := groupsByName["svc_dist"]; ok {
			_ = g
		}
	}
}
