package storage

import (
	"context"
	"io"
	"os"
	"path"
	"path/filepath"
	"strings"
	"testing"
	"time"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/config"
	"github.com/stretchr/testify/assert"
)

func setUpCurrent(p string) error {
	vPath, err := filepath.Abs(p)
	if err != nil {
		return err
	}
	curPath, err := filepath.Abs("repo/current")
	if err != nil {
		return err
	}
	return os.Symlink(vPath, curPath)
}

func TestTimestampRepo_CurrentVersion(t *testing.T) {
	name := path.Join(repoPath, "dummy")
	err := setUp(name, "")
	assert.NoError(t, err)
	err = setUpCurrent(repoPath)
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	r, err := NewTimestampRepo(&config.RepoConfig{Path: "repo"})
	assert.NoError(t, err)
	v, err := r.CurrentVersion()
	assert.NoError(t, err)
	expectedTS, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	assert.Equal(t, expectedTS, v.Timestamp())
}

func TestTimestampRepo_UpdateCurrent(t *testing.T) {
	name := path.Join(repoPath, "dummy")
	err := setUp(name, "")
	assert.NoError(t, err)
	err = setUpCurrent(repoPath)
	assert.NoError(t, err)
	name = "repo/1631799540/dummy"
	err = setUp(name, "")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	v, err := NewLocalVersion(path.Dir(name))
	assert.NoError(t, err)
	r, err := NewTimestampRepo(&config.RepoConfig{Path: "repo"})
	assert.NoError(t, err)
	oldCurrent, err := r.CurrentVersion()
	assert.NoError(t, err)
	err = r.UpdateCurrent(v)
	assert.NoError(t, err)
	newCurrent, err := r.CurrentVersion()
	assert.NoError(t, err)
	assert.NotEqual(t, newCurrent.Timestamp(), oldCurrent.Timestamp())
}

func TestTimestampRepo_GcRevisions(t *testing.T) {
	err := setUp("repo/1631799540/dummy", "")
	assert.NoError(t, err)
	err = setUp("repo/1631799541/dummy", "")
	assert.NoError(t, err)
	err = setUp("repo/1631799542/dummy", "")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	r, err := NewTimestampRepo(&config.RepoConfig{Path: "repo", KeepVersions: 2})
	assert.NoError(t, err)
	err = r.gcVersions()
	assert.NoError(t, err)
	entries, err := os.ReadDir("repo")
	assert.NoError(t, err)
	var dirs []string
	for _, ent := range entries {
		if ent.IsDir() && versionRe.MatchString(ent.Name()) {
			dirs = append(dirs, ent.Name())
		}
	}
	assert.Equal(t, []string{"1631799541", "1631799542"}, dirs)
}

type vMock struct {
	ts time.Time
}

func (v *vMock) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("warwish:*:53523:239396:warwish:/home/warwish:/bin/bash")}, nil
}

func (v *vMock) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("{\n  \"insecure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/insecure\",\n  \"sudo_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/sudo\",\n  \"krl_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/krl/all.zst\",\n  \"key_sources\": [\n    \"staff\"\n  ],\n  \"secure_ca_list_url\": \"https://skotty.sec.yandex-team.ru/api/v1/ca/pub/secure\"\n}")}, nil
}

func (v *vMock) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	return &readCloser{r: strings.NewReader("mock")}, nil
}

func (v *vMock) Timestamp() time.Time {
	return v.ts
}

type readCloser struct {
	r io.Reader
}

func (r *readCloser) Read(p []byte) (n int, err error) {
	return r.r.Read(p)
}

func (r *readCloser) Close() error {
	return nil
}

func TestTimestampRepo_PersistVersion(t *testing.T) {
	ts, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	v := &vMock{ts: ts}
	r, err := NewTimestampRepo(&config.RepoConfig{Path: "repo", KeepVersions: 2})
	assert.NoError(t, err)
	err = r.PersistVersion(context.TODO(), v)
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	entries, err := os.ReadDir("repo")
	assert.NoError(t, err)
	assert.Equal(t, 1, len(entries))
	s, err := os.Stat(repoPath)
	assert.NoError(t, err)
	assert.Equal(t, true, s.IsDir())
	assert.Equal(t, "1631799539", s.Name())
}

func assertFileContains(t *testing.T, file, expected string) {
	f, err := os.Open(file)
	assert.NoError(t, err)
	defer f.Close()
	buf, err := io.ReadAll(f)
	assert.NoError(t, err)
	assert.Equal(t, []byte(expected), buf)
}

func TestTimestampRepo_PersistVersion_AdminSudoers(t *testing.T) {
	ts, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	v := &vMock{ts: ts}
	r, err := NewTimestampRepo(&config.RepoConfig{Path: "repo", KeepVersions: 2})
	assert.NoError(t, err)
	err = r.PersistVersion(context.TODO(), v)
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	entries, err := os.ReadDir("repo")
	assert.NoError(t, err)
	assert.Equal(t, 1, len(entries))
	s, err := os.Stat(repoPath)
	assert.NoError(t, err)
	assert.Equal(t, true, s.IsDir())
	assert.Equal(t, "1631799539", s.Name())
	assertFileContains(t, path.Join(repoPath, adminSudoersFile), "warwish ALL=(ALL) NOPASSWD: ALL\n")
}
