package storage

import (
	"context"
	"io"
	"os"
	"path"
	"strconv"
	"time"

	"a.yandex-team.ru/infra/cauth/agent/linux/yandex-cauth-userd/internal/cauth"
)

const (
	// Be careful changing path constants
	// Some files referred outside code
	passwdFile       = "passwd"
	groupFile        = "group"
	accessFile       = "access.conf"
	adminUsersFile   = "serveradmins"
	keysFile         = "keys"
	sudoersFile      = "sudoers"
	adminSudoersFile = "admin-sudoers"
	keysInfoFile     = "keysinfo"
	insecureFile     = "insecure_ca_list"
	secureFile       = "secure_ca_list"
	sudoersCAFile    = "sudoers_ca_list"
	KRLFile          = "krl"
)

// Version encapsulates raw data CAuth snapshot access
type Version interface {
	cauth.Client
	Timestamp() time.Time
}

// localTextVer implements Version storing data in local text files.
type localTextVer struct {
	path string
	ts   time.Time
}

func NewLocalVersion(p string) (*localTextVer, error) {
	ts, err := strconv.ParseInt(path.Base(p), 10, 64)
	if err != nil {
		return nil, err
	}
	return &localTextVer{
		path: p,
		ts:   time.Unix(ts, 0),
	}, nil
}

func (v *localTextVer) open(name string) (io.ReadCloser, error) {
	f, err := os.Open(path.Join(v.path, name))
	if err != nil {
		return nil, err
	}
	return f, nil
}

func (v *localTextVer) FetchPasswd(_ context.Context) (io.ReadCloser, error) {
	return v.open(passwdFile)
}

func (v *localTextVer) FetchGroup(_ context.Context) (io.ReadCloser, error) {
	return v.open(groupFile)
}

func (v *localTextVer) FetchAccess(_ context.Context) (io.ReadCloser, error) {
	return v.open(accessFile)
}

func (v *localTextVer) FetchKeys(_ context.Context) (io.ReadCloser, error) {
	return v.open(keysFile)
}

func (v *localTextVer) FetchAdminUsers(_ context.Context) (io.ReadCloser, error) {
	return v.open(adminUsersFile)
}

func (v *localTextVer) FetchSudoers(_ context.Context) (io.ReadCloser, error) {
	return v.open(sudoersFile)
}

func (v *localTextVer) FetchKeysInfo(_ context.Context) (io.ReadCloser, error) {
	return v.open(keysInfoFile)
}

func (v *localTextVer) FetchInsecure(_ context.Context) (io.ReadCloser, error) {
	return v.open(insecureFile)
}

func (v *localTextVer) FetchSecure(_ context.Context) (io.ReadCloser, error) {
	return v.open(secureFile)
}

func (v *localTextVer) FetchSudo(_ context.Context) (io.ReadCloser, error) {
	return v.open(sudoersCAFile)
}

func (v *localTextVer) FetchKRL(_ context.Context) (io.ReadCloser, error) {
	return v.open(KRLFile)
}

func (v *localTextVer) Timestamp() time.Time {
	return v.ts
}

// remoteCauthVer implements Version for data retrieval from CAuth client.
type remoteCauthVer struct {
	c  cauth.Client
	ts time.Time
}

func (v *remoteCauthVer) FetchPasswd(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchPasswd(ctx)
}

func (v *remoteCauthVer) FetchGroup(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchGroup(ctx)
}

func (v *remoteCauthVer) FetchAccess(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchAccess(ctx)
}

func (v *remoteCauthVer) FetchKeys(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchKeys(ctx)
}

func (v *remoteCauthVer) FetchAdminUsers(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchAdminUsers(ctx)
}

func (v *remoteCauthVer) FetchSudoers(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchSudoers(ctx)
}

func (v *remoteCauthVer) FetchKeysInfo(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchKeysInfo(ctx)
}

func (v *remoteCauthVer) FetchInsecure(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchInsecure(ctx)
}

func (v *remoteCauthVer) FetchSecure(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchSecure(ctx)
}

func (v *remoteCauthVer) FetchSudo(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchSudo(ctx)
}

func (v *remoteCauthVer) FetchKRL(ctx context.Context) (io.ReadCloser, error) {
	return v.c.FetchKRL(ctx)
}

// Timestamp returns time.Now() of first call
// cauth server does not have any versions/revisions, so we're using local timestamps
// which will be different on each host
func (v *remoteCauthVer) Timestamp() time.Time {
	if v.ts.IsZero() {
		v.ts = time.Now()
	}
	return v.ts
}

func NewRemoteVersion(c cauth.Client) *remoteCauthVer {
	return &remoteCauthVer{c: c}
}
