package storage

import (
	"context"
	"io"
	"os"
	"path"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
)

const (
	repoPath = "repo/1631799539"
	repoTime = "Thu Sep 16 16:38:59 MSK 2021"
)

func TestNewLocalVersion(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	expected, err := time.Parse(time.UnixDate, repoTime)
	assert.NoError(t, err)
	assert.Equal(t, expected, v.Timestamp())
}

func TestRemoteCauthVer_Timestamp(t *testing.T) {
	v := NewRemoteVersion(nil)
	t1 := v.Timestamp()
	t2 := v.Timestamp()
	assert.Equal(t, t1, t2)
}

func setUp(name, content string) error {
	return setUpAll([]*file{{name: name, content: content}})
}

type file struct {
	name    string
	content string
}

func setUpAll(files []*file) error {
	err := os.MkdirAll(path.Dir(files[0].name), os.ModeDir|0o755)
	if err != nil {
		return err
	}
	for _, file := range files {
		f, err := os.OpenFile(file.name, os.O_CREATE|os.O_RDWR|os.O_TRUNC, 0o644)
		if err != nil {
			return err
		}
		defer f.Close()
		_, err = f.WriteString(file.content)
		if err != nil {
			return err
		}
	}
	return err
}

func tearDown() error {
	return os.RemoveAll("repo")
}

func TestLocalTextVer_FetchAccess(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, accessFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchAccess(context.TODO())
	assert.NoError(t, err)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}

func TestLocalTextVer_FetchGroup(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, groupFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchGroup(context.TODO())
	assert.NoError(t, err)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}

func TestLocalTextVer_FetchAdminUsers(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, adminUsersFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchAdminUsers(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, adminUsersFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}

func TestLocalTextVer_FetchKeys(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, keysFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchKeys(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, keysFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}

func TestLocalTextVer_FetchPasswd(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, passwdFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchPasswd(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, passwdFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}

func TestLocalTextVer_FetchKeysInfo(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, keysInfoFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchKeysInfo(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, keysInfoFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}
func TestLocalTextVer_FetchInsecure(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, insecureFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchInsecure(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, insecureFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}
func TestLocalTextVer_FetchSecure(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, secureFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchSecure(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, secureFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}
func TestLocalTextVer_FetchSudo(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, sudoersCAFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchSudo(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, sudoersCAFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}
func TestLocalTextVer_FetchKRL(t *testing.T) {
	v, err := NewLocalVersion(repoPath)
	assert.NoError(t, err)
	file := path.Join(repoPath, KRLFile)
	err = setUp(file, "mock")
	assert.NoError(t, err)
	defer func() { assert.NoError(t, tearDown()) }()
	result, err := v.FetchKRL(context.TODO())
	assert.NoError(t, err)
	assert.Equal(t, path.Join(repoPath, KRLFile), file)
	b, err := io.ReadAll(result)
	assert.NoError(t, err)
	assert.Equal(t, "mock", string(b))
}
