#include "bench_pwd.h"
#include "bench_grp.h"

template <auto F, typename FArg>
void benchmark(std::string name, long threads, long requests, BenchGen<FArg>* gen, double releaseFactor) {
    auto nssBench = Bench<F, FArg>(name, threads, requests, gen, releaseFactor);
    auto start = std::chrono::high_resolution_clock::now();
    nssBench.bench();
    auto end = std::chrono::high_resolution_clock::now();
    nssBench.gather_results();
    nssBench.print_results(std::cout);
    std::chrono::nanoseconds diff = end - start;
    std::cout << "time elapsed: " << diff.count() / 1000 << "us" << std::endl;
    std::cout << "average rps: " << double(requests) / double(diff.count()) * 1000000000 << std::endl;
}

void run_getpwnam(long threads, long requests, double releaseFactor) {
    auto gen = LoginBenchGen(USERS);
    benchmark<bench_getpwnam, std::string>("getpwnam", threads, requests, &gen, releaseFactor);
}

void run_getpwnam_r(long threads, long requests, double releaseFactor) {
    auto gen = LoginBenchGen(USERS);
    benchmark<bench_getpwnam_r, std::string>("getpwnam_r", threads, requests, &gen, releaseFactor);
}

void run_getgrnam(long threads, long requests, double releaseFactor) {
    auto gen = GrNameBenchGen(GROUPS);
    benchmark<bench_getgrnam, std::string>("getgrnam", threads, requests, &gen, releaseFactor);
}

void run_getgrnam_r(long threads, long requests, double releaseFactor) {
    auto gen = GrNameBenchGen(GROUPS);
    benchmark<bench_getgrnam_r, std::string>("getgrnam_r", threads, requests, &gen, releaseFactor);
}

void run_getpwuid(long threads, long requests, double releaseFactor) {
    auto gen = UidBenchGen(USERS);
    benchmark<bench_getpwuid, uid_t>("getpwuid", threads, requests, &gen, releaseFactor);
}

void run_getpwuid_r(long threads, long requests, double releaseFactor) {
    auto gen = UidBenchGen(USERS);
    benchmark<bench_getpwuid_r, uid_t>("getpwuid_r", threads, requests, &gen, releaseFactor);
}

void run_getgrgid(long threads, long requests, double releaseFactor) {
    auto gen = GidBenchGen(GROUPS);
    benchmark<bench_getgrgid, gid_t>("getgrgid", threads, requests, &gen, releaseFactor);
}

void run_getgrgid_r(long threads, long requests, double releaseFactor) {
    auto gen = GidBenchGen(GROUPS);
    benchmark<bench_getgrgid_r, gid_t>("getgrgid_r", threads, requests, &gen, releaseFactor);
}

const std::map<std::string, void (*)(long, long, double)> BENCHMARKS = {
    {"getpwnam", &run_getpwnam},
    {"getpwnam_r", &run_getpwnam_r},
    {"getgrnam", &run_getgrnam},
    {"getgrnam_r", &run_getgrnam_r},
    {"getpwuid", &run_getpwuid},
    {"getpwuid_r", &run_getpwuid_r},
    {"getgrgid", &run_getgrgid},
    {"getgrgid_r", &run_getgrgid_r},
};

void usage(char* argv0) {
    std::cout << "usage: " << argv0 << " <bench> <threads> <requests> <sem_release_factor>" << std::endl;
    std::cout << "available benchmarks:" << std::endl;
    for (const auto& b : BENCHMARKS) {
        std::cout << b.first << std::endl;
    }
}

int main(int argc, char* argv[]) {
    if (argc != 5) {
        usage(argv[0]);
        return 1;
    }
    std::string bench(argv[1]);
    if (!BENCHMARKS.contains(bench)) {
        usage(argv[0]);
        return 1;
    }
    auto nThreads = atol(argv[2]);
    auto nRequests = atol(argv[3]);
    auto factor = atof(argv[4]);
    BENCHMARKS.at(bench)(nThreads, nRequests, factor);
    return 0;
}
