#!/bin/bash

checklog="/var/log/auth.log"
# monitoring log
log="/var/log/saslcheck/saslcheck.log"
err_search="ldap_simple_bind() failed -1"
default="/etc/default/saslauthd"
# number of seconds to watch back the log file
interval=60
tail="/usr/bin/timetail -t syslog -n $interval "
# found errors limit for rising critical error (ldap_simple_bind() failed -1)
conn_err_limit=2

wrlog()
{
	echo `date -R`" saslauthd_check" "$1" >> $log
	if [ $? -ne 0 ]
	then
		return 1
	else
		return 0
	fi
}

ldapsconncheck()
{
    err_found=`$tail $checklog | grep -i "$err_search" | wc -l`

	if [ $err_found -gt 0 ] && [ $err_found -lt $conn_err_limit ]
	then
		wrlog "Warning: Detected $err_found connection errors for last $interval seconds."
		echo 1
		return 0
	fi

    if [ $err_found -ge $conn_err_limit ]
    then
		wrlog "Critical: Detected $err_found connection errors for last $interval seconds."
        echo 2
		return 0
    fi

	echo 0
	return 0
}

saslauthdisalive()
{
    if [ -f $default ]
    then
        runpath=`grep -E "^OPTIONS" $default | grep "\-m" | awk -F "-m" '{print $2;}' | awk '{print $1}'`
        name=`grep -E "^NAME" $default | awk -F "=" '{print $2}' | tr -d "\""`

        if [ ${#name} -le 1 ] || [ ${#runpath} -le 1 ]
        then
            echo 2
	    	return 0
        fi

        if [ ! -d $runpath ]
		then
			echo 3
			return 0
		else
			if [ ! -f $runpath/$name.pid  ]
			then
		    	echo 5
		    	return 0
        	else
		    	pid=`cat $runpath/$name.pid`
            fi
        fi

        isalive=`ps auwx | grep $pid | grep -v grep | wc -l`

        if [ $isalive -lt 1 ]
        then
		    echo 6
		    return 0
		else
		    echo 0
	    	return 0
		fi
    else
        echo 4
		return 0
    fi
    return 0
}

ld_check=$(ldapsconncheck)
sa_check=$(saslauthdisalive)

# define monrun message

if [ $ld_check -eq 0 ]
then
    ld_msg="OK"
elif [ $ld_check -eq 1 ]
then
	ld_msg="Warning. ldaps detected connection errors, but it's number is lower than $conn_err_limit"
else
    ld_msg="Critical. ldaps detected connection errors more than $conn_err_limit"
fi

case $sa_check in
    0 )
	sa_msg="OK"
    ;;
    2 )
        sa_msg="Critical. Instance name or runpath detection error($default)."
    ;;
    3 )
        sa_msg="Critical. saslauthd run directory not found(see OPTIONS in /etc/default/saslauthd)"
    ;;
    4 )
	sa_msg="Critical. $default config not found"
    ;;
    5 )
	sa_msg="Critical. saslauthd pid file not found"
    ;;
    6 )
	sa_msg="Critical. saslauthd is not running"
    ;;
    * )
	sa_msg="Critical. some unknown error"
    ;;
esac

err_sum=$(($ld_check + $sa_check))

if  [ $err_sum -eq 0 ]
then
	echo "0;ldaps_conn: $ld_msg;saslauthd_alive: $sa_msg"
elif [ $err_sum -eq 1 ]
then
	echo "1;ldaps_conn: $ld_msg;saslauthd_alive: $sa_msg"
elif [ $err_sum -eq 2 ] && [ $ld_check -eq 1 ] && [ $sa_check -eq 1 ]
then
	echo "1;ldaps_conn: $ld_msg;saslauthd_alive: $sa_msg"
else
	echo "2;ldaps_conn: $ld_msg;saslauthd_alive: $sa_msg"
fi

exit 0

