import datetime

from django.conf import settings
from django.utils.functional import cached_property
from sqlalchemy import (Table, Column, ForeignKey, Index, Integer, String, Boolean, Enum,
                        Text, DateTime, Date, PrimaryKeyConstraint, or_, UniqueConstraint)
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import relationship, configure_mappers, backref

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.constants import (
    CLIENT_SOURCE_REASON, ACTION_TYPE, USER_GROUP_TYPE, SERVER_TYPE, IDM_STATUS, ACCESS_TYPE,
    FLOW_TYPE, KEY_SOURCE, KEYS_INFO_ATTR
)


class InnodbBase:

    query = Session.query_property()

    @classmethod
    def get_one(cls, **kwargs):
        return cls.query.filter_by(**kwargs).one()

    def __str__(self):
        if hasattr(self, '__unicode__'):
            return self.__unicode__()
        super(InnodbBase, self).__str__()

    def __repr__(self):
        if hasattr(self, '__unicode__'):
            return '<{}: {}>'.format(self.__class__.__name__, self.__unicode__())
        return super(InnodbBase, self).__repr__()


BaseModel = declarative_base(cls=InnodbBase)


class UserGroupRelation(BaseModel):
    __tablename__ = 'new_users_m2m_groups'
    __table_args__ = (PrimaryKeyConstraint('uid', 'gid'),)

    uid = Column(
        Integer,
        ForeignKey('new_users.uid', ondelete='CASCADE'),
    )
    gid = Column(
        Integer,
        ForeignKey('new_groups.gid', ondelete='CASCADE'),
        index=True,
    )
    until = Column(Date, default=None)

    user = relationship('User')
    group = relationship('Group')

    def __unicode__(self):
        return 'uid:{s.uid} in gid:{s.gid} until {s.until}'.format(s=self)

    @property
    def is_expired(self):
        return self.until is not None and self.until <= datetime.date.today()

    @staticmethod
    def add_active_filter(query):
        return query.filter(
            or_(
                UserGroupRelation.until.is_(None),
                UserGroupRelation.until > datetime.date.today(),
            )
        )

    @classmethod
    def get_active_query(cls):
        return cls.add_active_filter(cls.query)


class Group(BaseModel):
    __tablename__ = 'new_groups'

    gid = Column(Integer, primary_key=True)
    parent_gid = Column(Integer)
    type = Column(
        Enum(*USER_GROUP_TYPE.choises(), name='user_group_type'),
        nullable=False,
    )
    name = Column(String(128), unique=True, nullable=False)
    service_id = Column(Integer)
    staff_id = Column(Integer)

    parent = relationship(
        'Group',
        primaryjoin='Group.parent_gid==Group.gid',
        remote_side=gid,
        foreign_keys='Group.parent_gid'
    )

    def __unicode__(self):
        return self.name


Index('groups_gid_name', Group.gid, Group.name)


def user_home_default(context):
    return '/home/' + context.current_parameters['login']


class User(BaseModel):
    __tablename__ = 'new_users'

    uid = Column(Integer, primary_key=True)
    gid = Column(Integer, nullable=False)
    login = Column(String(128), unique=True, nullable=False)
    first_name = Column(String(50), nullable=False)
    last_name = Column(String(100))
    join_date = Column(Date)
    shell = Column(String(50), nullable=False)
    home = Column(String(255), nullable=False, default=user_home_default)
    is_fired = Column(Boolean, default=False, nullable=False)
    is_robot = Column(Boolean, default=False, nullable=False)

    @property
    def email(self):
        return '@'.join((self.login, 'yandex-team.ru'))

    groups = relationship(
        Group,
        secondary=UserGroupRelation.__table__,
        passive_deletes=True,
        backref=backref('users', passive_deletes=True),
    )

    def __unicode__(self):
        return self.login


sg_m2m = Table(
    'new_servers_m2m_groups',
    BaseModel.metadata,

    Column(
        'server_id',
        Integer,
        ForeignKey('new_servers.id', ondelete='CASCADE'),
        primary_key=True,
    ),
    Column(
        'group_id', Integer,
        ForeignKey('new_server_groups.id', ondelete='CASCADE'),
        primary_key=True,
        index=True,
    ),
)


gr_m2m = Table(
    'new_server_group_responsibles',
    BaseModel.metadata,

    Column(
        'group_id',
        Integer,
        ForeignKey('new_server_groups.id', ondelete='CASCADE'),
        primary_key=True,
    ),
    Column(
        'uid',
        Integer,
        ForeignKey('new_users.uid', ondelete='CASCADE'),
        primary_key=True,
        index=True
    ),
)


class Source(BaseModel):
    """Когда сервер или ответственный пушится из источника, ему назначается
    источник default. При следующей синхронизации он должен замениться на
    нормальный. Сервер может принадлежать нескольким источникам. Ответсвенные за
    сервер так же могут приезжать из нескольких источников. Если сервер
    принадлежит источнику default, то он не должен принадлежать другим
    источникам. Если сервер долго находится в default и не резолвится через dns,
    он должен удаляться."""
    __tablename__ = 'new_sources'

    id = Column(Integer, primary_key=True)
    name = Column(String(16), unique=True, nullable=False)
    is_default = Column(Boolean, nullable=False, default=False)
    is_modern = Column(Boolean, nullable=False, default=False)
    last_update = Column(DateTime)

    def __unicode__(self):
        return self.name


server_source_m2m = Table(
    'new_server_sources',
    BaseModel.metadata,

    Column(
        'server_id',
        Integer,
        ForeignKey('new_servers.id', ondelete='CASCADE'),
        primary_key=True
    ),
    Column(
        'source_id',
        Integer,
        ForeignKey('new_sources.id', ondelete='CASCADE'),
        primary_key=True,
        index=True
    ),
)


class ServerGroup(BaseModel):
    __tablename__ = 'new_server_groups'

    id = Column(Integer, primary_key=True)
    name = Column(String(128), unique=True, nullable=False)
    email = Column(String(128))
    notify_email = Column(String(128))
    request_queue = Column(String(128))
    source_id = Column(
        Integer,
        ForeignKey('new_sources.id', ondelete='SET NULL'),
    )
    idm_status = Column(
        Enum(*IDM_STATUS.choices(), name='idm_status'),
        default=IDM_STATUS.ACTUAL,
        nullable=False,
    )

    created_at = Column(DateTime, default=datetime.datetime.utcnow, nullable=True)
    updated_at = Column(DateTime, onupdate=datetime.datetime.utcnow, nullable=True)

    last_push_ended_at = Column(DateTime, default=None, nullable=True)
    first_pending_push_started_at = Column(DateTime, default=None, nullable=True)
    became_dirty_at = Column(DateTime, default=None, nullable=True)
    became_actual_at = Column(DateTime, default=None, nullable=True)

    responsible_users = relationship(
        User,
        secondary=gr_m2m,
        passive_deletes=True,
        backref=backref('responsible_for_groups', passive_deletes=True)
    )
    source = relationship(
        Source,
        backref=backref('groups', passive_deletes=True),
    )
    flow = Column(
        Enum(*FLOW_TYPE.choices(), name='flow_type'),
        default=FLOW_TYPE.CLASSIC,
        nullable=False,
    )
    trusted_sources = relationship(
        Source,
        secondary=lambda: ServerGroupTrustedSourceRelation.__table__,
        passive_deletes=True,
        backref=backref('trusted_servergroups', passive_deletes=True),
    )
    key_sources = Column(String(128), default=None, nullable=True)
    secure_ca_list_url = Column(String(128), default=None, nullable=True)
    insecure_ca_list_url = Column(String(128), default=None, nullable=True)
    krl_url = Column(String(128), default=None, nullable=True)
    sudo_ca_list_url = Column(String(128), default=None, nullable=True)

    def __unicode__(self):
        return self.name


class Server(BaseModel):
    __tablename__ = 'new_servers'

    id = Column(Integer, primary_key=True)
    fqdn = Column(String(128), unique=True, nullable=False)
    client_version = Column(String(50))
    idm_status = Column(
        Enum(*IDM_STATUS.choices(), name='idm_status'),
        default=IDM_STATUS.ACTUAL,
        nullable=False,
    )
    created_at = Column(DateTime, default=datetime.datetime.utcnow, nullable=True)
    updated_at = Column(DateTime, onupdate=datetime.datetime.utcnow, nullable=True)

    last_push_ended_at = Column(DateTime, default=None, nullable=True)
    first_pending_push_started_at = Column(DateTime, default=None, nullable=True)
    became_dirty_at = Column(DateTime, default=None, nullable=True)
    became_actual_at = Column(DateTime, default=None, nullable=True)
    is_baremetal = Column(Boolean, default=False)

    type = Column(
        Enum(*SERVER_TYPE.choices(), name='server_type'),
        default=SERVER_TYPE.SERVER,
        nullable=False,
    )

    groups = relationship(
        ServerGroup,
        secondary=sg_m2m,
        passive_deletes=True,
        backref=backref('servers', passive_deletes=True),
    )

    responsible_users = relationship(
        User,
        passive_deletes=True,
        secondary=lambda: ServerResponsible.__table__,
        backref=backref('responsible_for_servers', passive_deletes=True),
    )

    sources = relationship(
        Source,
        secondary=server_source_m2m,
        passive_deletes=True,
        backref=backref('servers', passive_deletes=True),
    )

    classic_trusted_sources = relationship(
        Source,
        secondary=lambda: ServerTrustedSourceRelation.__table__,
        passive_deletes=True,
        backref=backref('trusted_servers', passive_deletes=True),
    )

    owner_id = Column(
        Integer,
        ForeignKey(Source.id, ondelete='RESTRICT'),
        nullable=True,
    )
    key_sources = Column(String(128), default=KEY_SOURCE.STAFF, nullable=False)
    secure_ca_list_url = Column(String(128), default=None, nullable=True)
    insecure_ca_list_url = Column(String(128), default=None, nullable=True)
    krl_url = Column(String(128), default=None, nullable=True)
    sudo_ca_list_url = Column(String(128), default=None, nullable=True)

    @cached_property
    def authoritative_group(self):
        for flow_source_name in settings.CAUTH_FLOW_SOURCES_BY_PRIO:
            flow_source_group = ServerGroup.query.join(sg_m2m).join(Source).filter(
                sg_m2m.c.server_id == self.id,
                Source.name == flow_source_name,
            ).first()
            if flow_source_group:
                return flow_source_group

    @cached_property
    def flow(self):
        if self.authoritative_group:
            return self.authoritative_group.flow
        return FLOW_TYPE.CLASSIC

    def set_flow(self, flow):
        # Сбросим закешированное значение
        if hasattr(self, 'authoritative_group'):
            delattr(self, 'authoritative_group')
        if self.authoritative_group:
            self.authoritative_group.flow = flow
            return
        if flow == FLOW_TYPE.BACKEND_SOURCES:
            raise RuntimeError("flow type '{}' requires authoritative group existence".format(flow))

    @cached_property
    def trusted_sources(self):
        if self.flow == FLOW_TYPE.BACKEND_SOURCES:
            return self.authoritative_group.trusted_sources
        return self.classic_trusted_sources

    def __unicode__(self):
        return self.fqdn

    @cached_property
    def keys_info_group(self):
        for source in settings.CAUTH_KEYS_INFO_SOURCES_BY_PRIO:
            keys_info_group = ServerGroup.query.join(sg_m2m).join(Source).filter(
                sg_m2m.c.server_id == self.id,
                Source.name == source,
            ).first()
            if keys_info_group:
                return keys_info_group

    def get_keys_info(self, attr):
        return (
            getattr(self.keys_info_group, attr, None)
            or getattr(self, attr, None)
            or KEYS_INFO_ATTR.DEFAULTS.get(attr)
        )


class ServerResponsible(BaseModel):
    __tablename__ = 'new_server_responsibles'

    server_id = Column(
        Integer,
        ForeignKey('new_servers.id', ondelete='CASCADE'),
        primary_key=True,
    )
    uid = Column(
        Integer,
        ForeignKey('new_users.uid', ondelete='CASCADE'),
        primary_key=True,
        index=True,
    )
    source_id = Column(
        Integer,
        ForeignKey('new_sources.id', ondelete='CASCADE'),
        primary_key=True,
    )

    server = relationship(
        Server,
        backref=backref(
            'responsibles',
            cascade='all, delete-orphan',
            passive_deletes=True,
        ))
    user = relationship(
        User,
        backref=backref(
            'server_responsibilities',
            cascade='all',
            passive_deletes=True,
        )
    )
    source = relationship(
        Source,
        backref=backref(
            'server_responsibilities',
            cascade='all',
            passive_deletes=True,
        )
    )

    def __unicode__(self):
        return '{0.user.login} -> {0.server.fqdn} (from {0.source.name})'.format(self)


class ServerTrustedSourceRelation(BaseModel):
    __tablename__ = 'new_servers_m2m_trusted_sources'
    __table_args__ = (
        UniqueConstraint('server_id', 'source_id', name='server_source_uc'),
    )

    id = Column(Integer, primary_key=True)
    server_id = Column(Integer, ForeignKey(Server.id, ondelete='CASCADE'))
    source_id = Column(Integer, ForeignKey(Source.id, ondelete='CASCADE'))
    created = Column(DateTime, default=datetime.datetime.utcnow, nullable=False)
    reason = Column(
        Enum(*CLIENT_SOURCE_REASON.choices(), name='client_source_reason'),
        nullable=False,
    )

    server = relationship(Server)
    source = relationship(Source)

    @classmethod
    def create(cls, session, server_id, source_id, reason):
        trusted_source = cls(
            server_id=server_id,
            source_id=source_id,
            reason=reason,
        )
        add_action = Action(
            type=ACTION_TYPE.ADD_TRUSTED_SOURCE,
            server_id=server_id,
            source_id=source_id,
        )
        session.add_all((trusted_source, add_action))

    def delete(self, session):
        remove_action = Action(
            type=ACTION_TYPE.REMOVE_TRUSTED_SOURCE,
            server_id=self.server_id,
            source_id=self.source_id,
        )
        session.add(remove_action)

        session.delete(self)

    def __unicode__(self):
        return 'server_id:{s.server_id} <-> {s.source_id}'.format(s=self)


class ServerGroupTrustedSourceRelation(BaseModel):
    __tablename__ = 'new_servergroups_m2m_trusted_sources'
    __table_args__ = (
        UniqueConstraint('servergroup_id', 'source_id', name='servergroup_source_uc'),
    )

    id = Column(Integer, primary_key=True)
    servergroup_id = Column(Integer, ForeignKey(ServerGroup.id, ondelete='CASCADE'))
    source_id = Column(Integer, ForeignKey(Source.id, ondelete='CASCADE'))
    created = Column(DateTime, default=datetime.datetime.utcnow, nullable=False)
    reason = Column(
        Enum(*CLIENT_SOURCE_REASON.choices(), name='client_source_reason'),
        nullable=False,
    )

    servergroup = relationship(ServerGroup)
    source = relationship(Source)

    @classmethod
    def create(cls, session, servergroup_id, source_id, reason):
        trusted_source = cls(
            servergroup_id=servergroup_id,
            source_id=source_id,
            reason=reason,
        )
        add_action = Action(
            type=ACTION_TYPE.ADD_TRUSTED_SOURCE,
            servergroup_id=servergroup_id,
            source_id=source_id,
        )
        session.add_all((trusted_source, add_action))

    def delete(self, session):
        remove_action = Action(
            type=ACTION_TYPE.REMOVE_TRUSTED_SOURCE,
            servergroup_id=self.servergroup_id,
            source_id=self.source_id,
        )
        session.add(remove_action)

        session.delete(self)

    def __unicode__(self):
        return 'servergroup_id:{s.servergroup_id} <-> {s.source_id}'.format(s=self)


class Action(BaseModel):
    __tablename__ = 'actions'

    id = Column(Integer, primary_key=True)
    type = Column(
        Enum(*ACTION_TYPE.choices(), name='action_type'),
        nullable=False,
    )
    server_id = Column(Integer, ForeignKey(Server.id, ondelete='CASCADE'))
    servergroup_id = Column(Integer, ForeignKey(ServerGroup.id, ondelete='CASCADE'))
    source_id = Column(Integer, ForeignKey(Source.id, ondelete='CASCADE'))
    created = Column(DateTime, default=datetime.datetime.utcnow, nullable=False)

    servers = relationship(
        Server,
        backref=backref('actions', passive_deletes=True),
    )

    servergroups = relationship(
        ServerGroup,
        backref=backref('actions', passive_deletes=True),
    )

    source = relationship(
        Source,
        backref=backref('actions', passive_deletes=True),
    )

    def __unicode__(self):
        return (
            '{s.type}: '
            'server:{s.server_id}, '
            'servergroup:{s.servergroup_id}, '
            'source:{s.source_id}'
        ).format(s=self)


class PublicKey(BaseModel):
    __tablename__ = 'new_pubkeys'
    id = Column(Integer, primary_key=True)
    uid = Column(
        Integer,
        ForeignKey('new_users.uid', ondelete='CASCADE'),
        nullable=False,
    )
    key = Column(Text, nullable=False)
    user = relationship(
        User,
        backref=backref(
            'public_keys',
            cascade='all, delete-orphan',
            passive_deletes=True,
        )
    )

    def __unicode__(self):
        return 'Public key for {}'.format(self.user.login)


class Role(BaseModel):
    __tablename__ = 'new_roles'
    id = Column(Integer, primary_key=True)
    spec = Column(String(1000))

    def __unicode__(self):
        return self.spec


class Access(BaseModel):
    __tablename__ = 'new_access'

    id = Column(Integer, primary_key=True)
    old_id = Column(Integer, index=True)

    type = Column(
        Enum(*ACCESS_TYPE.choices(), name='access_type'),
        nullable=False,
        index=True,
    )

    src = Column(String(128), nullable=False)
    src_user_id = Column(
        Integer,
        ForeignKey('new_users.uid', ondelete='SET NULL'),
    )
    src_group_id = Column(
        Integer,
        ForeignKey('new_groups.gid', ondelete='SET NULL'),
    )

    dst = Column(String(128), nullable=False)
    dst_server_id = Column(
        Integer,
        ForeignKey('new_servers.id', ondelete='SET NULL'),
    )
    dst_group_id = Column(
        Integer,
        ForeignKey('new_server_groups.id', ondelete='SET NULL'),
    )

    requester = Column(String(128), nullable=False)
    request_date = Column(DateTime, nullable=False)
    approver = Column(String(128), default=None)
    approve_date = Column(DateTime, default=None)

    description = Column(Text, nullable=False)
    until = Column(Date, default=None)

    sudo_role_id = Column(
        Integer,
        ForeignKey('new_roles.id', ondelete='SET NULL'),
        nullable=True,
    )
    ssh_is_admin = Column(Boolean, default=False, nullable=False)

    src_user = relationship(
        User,
        backref=backref('access_rules', passive_deletes=True),
    )
    src_group = relationship(
        Group,
        backref=backref('access_rules', passive_deletes=True),
    )
    dst_server = relationship(
        Server,
        backref=backref('access_rules', passive_deletes=True),
    )
    dst_group = relationship(
        ServerGroup,
        backref=backref('access_rules', passive_deletes=True),
    )
    sudo_role = relationship(
        Role,
        backref=backref('access_rules', passive_deletes=True),
    )

    def __unicode__(self):
        type_ext = ''
        if self.type == 'ssh' and self.ssh_is_admin:
            type_ext = '; root=True'
        elif self.type == 'sudo':
            type_ext = '; sudo_spec="{}"'.format(self.sudo_role.spec)

        return '{} -({})-> {}{}'.format(self.src, self.type, self.dst, type_ext)

    @property
    def is_expired(self):
        return self.until is not None and self.until <= datetime.date.today()

    @staticmethod
    def add_active_filter(query):
        return query.filter(
            Access.approver.isnot(None),
            Access.approve_date.isnot(None),
            or_(
                Access.until.is_(None),
                Access.until > datetime.date.today(),
            ),
            or_(
                Access.src_user_id.isnot(None),
                Access.src_group_id.isnot(None),
            ),
            or_(
                Access.dst_server_id.isnot(None),
                Access.dst_group_id.isnot(None),
            ),
        )

    @staticmethod
    def exclude_empty_dst(query):
        return (
            query
            .outerjoin(Access.dst_group)
            .outerjoin(ServerGroup.servers)
            .filter(
                or_(
                    Access.dst_server_id.isnot(None),
                    Server.id.isnot(None),
                )
            )
            .distinct()
        )

    @classmethod
    def get_active_query(cls):
        query_method = cls.add_active_filter(cls.query)
        return cls.exclude_empty_dst(query_method)

    @classmethod
    def get_active_query_with_empty_dst(cls):
        return cls.add_active_filter(cls.query)


Index('access_search_me', Access.src_user_id, Access.src_group_id)
Index('access_search_mine', Access.dst_server_id, Access.dst_group_id)


class ImportStatus(BaseModel):
    __tablename__ = 'import_status'

    suite = Column(String(16), primary_key=True)  # TODO: rename to group
    target = Column(String(16), primary_key=True)
    last_import = Column(DateTime, nullable=False)


class SkippedStaffInconsistencies(BaseModel):
    __tablename__ = 'skipped_inconsistency'

    id = Column(Integer, primary_key=True)
    error = Column(Text, nullable=False)


configure_mappers()
