import sys
import os

import jinja2

from django.template import TemplateDoesNotExist
from django.template.backends.django import copy_exception
from django.template.loaders.cached import Loader as CachedLoader
from django.utils.inspect import func_supports_parameter

from jinja2.utils import concat
from jinja2.exceptions import TemplateNotFound

from library.python.django.template.loaders.app_resource import Loader


class CauthCachedLoader(CachedLoader):
    def get_arcadia_template(self, template_name, template_dirs=None, skip=None):
        tried = []

        args = [template_name]
        if func_supports_parameter(self.get_template_sources, 'template_dirs'):
            args.append(template_dirs)

        for origin in self.get_template_sources(*args):
            if skip is not None and origin in skip:
                tried.append((origin, 'Skipped'))
                continue

            try:
                contents = self.get_contents(origin)
            except TemplateDoesNotExist:
                tried.append((origin, 'Source does not exist'))
                continue
            else:
                template = origin.loader.environment.from_string(contents, template_class=Template)
                template.origin = origin
                return template

        raise TemplateDoesNotExist(template_name, tried=tried)

    def get_template(self, template_name, template_dirs=None, skip=None):
        key = self.cache_key(template_name, template_dirs, skip)
        cached = self.get_template_cache.get(key)
        if cached:
            if isinstance(cached, type) and issubclass(cached, TemplateDoesNotExist):
                raise cached(template_name)
            elif isinstance(cached, TemplateDoesNotExist):
                raise copy_exception(cached)
            return cached

        try:
            template = self.get_arcadia_template(template_name, template_dirs, skip)
        except TemplateDoesNotExist as e:
            self.get_template_cache[key] = copy_exception(e) if self.engine.debug else TemplateDoesNotExist
            raise
        else:
            self.get_template_cache[key] = template

        return template


class AppDirJinja2Loader(Loader):
    def __init__(self, engine):
        super(AppDirJinja2Loader, self).__init__(engine)
        self.environment = jinja2.Environment(loader=JinjaAppDirLoader(self))


class JinjaAppDirLoader(jinja2.BaseLoader):
    def __init__(self, django_loader):
        self._django_loader = django_loader

    def get_source(self, environment, template):
        try:
            contents, filename = self._django_loader.load_template_source(template)
        except TemplateDoesNotExist as error:
            raise TemplateNotFound(str(error))

        mtime = os.path.getmtime(filename)

        def uptodate():
            try:
                return os.path.getmtime(filename) == mtime
            except OSError:
                return False
        return contents, filename, uptodate


class Template(jinja2.Template):
    def render(self, *args, **kwargs):
        vars = dict(*args, **kwargs)
        context = self.new_context(vars)

        try:
            return concat(self.root_render_func(context))
        except Exception:
            exc_info = sys.exc_info()

        return self.environment.handle_exception(str(exc_info))
