import re

from django import forms
from django.conf import settings
from django.core.validators import URLValidator

from infra.cauth.server.common.constants import SERVER_TYPE, FLOW_TYPE, KEY_SOURCE
from infra.cauth.server.common.models import User, Server, Source
from infra.cauth.server.master.api.fields import AlchemyFallbackChoiceField
from infra.cauth.server.master.utils.fqdn import should_be_added

ERR_INVALID_SERVER = 'Server not found.'


class NullableCharField(forms.CharField):
    def to_python(self, value):
        if value is None:
            return None
        return super(NullableCharField, self).to_python(value)


class AddServerForm(forms.Form):
    srv = forms.RegexField(max_length=128, regex=r'^[\w\.-]+$')
    grp = forms.CharField(required=False)
    resp = NullableCharField(required=False)
    type = forms.ChoiceField(choices=SERVER_TYPE.tuple_choices(), required=False)
    flow = forms.ChoiceField(choices=FLOW_TYPE.tuple_choices(), required=False)
    trusted_sources = forms.CharField(required=False)
    key_sources = forms.CharField(required=False)
    secure_ca_list_url = forms.CharField(required=False, validators=[URLValidator(schemes=['https', 'http'])])
    insecure_ca_list_url = forms.CharField(required=False, validators=[URLValidator(schemes=['https', 'http'])])
    krl_url = forms.CharField(required=False, validators=[URLValidator(schemes=['https', 'http'])])
    sudo_ca_list_url = forms.CharField(required=False, validators=[URLValidator(schemes=['https', 'http'])])

    def __init__(self, *args, **kwargs):
        from infra.cauth.server.master.importers.servers.aggregator import ServersGroupsAggregator
        group_prefixes = [
            source.PREFIX for source in ServersGroupsAggregator.group_sources
            if source.PREFIX in settings.GROUP_SOURCES
        ]

        re_template = r'^(?:%s)\.[\w\.-]+$'
        self.re_group = re.compile(re_template % '|'.join(group_prefixes))
        super(AddServerForm, self).__init__(*args, **kwargs)

    def clean_srv(self):
        srv = self.cleaned_data['srv']
        if not should_be_added(srv):
            self.add_error('srv', '{}: server fqdn is ignored in CAuth'.format(srv))
        return srv

    def clean_grp(self):
        grp = self.cleaned_data['grp']
        if not grp:
            return []

        group_names = grp.split(',')
        for name in group_names:
            if not self.re_group.match(name):
                self.add_error('grp', '{}: invalid group name'.format(name))

        return group_names

    def clean_resp(self):
        if 'resp' not in self.cleaned_data:
            return None
        data = self.cleaned_data['resp']
        if data is None:
            return data
        if not data:
            return []
        logins = {login.lower() for login in data.split(',')}
        users = User.query.filter(User.login.in_(logins)).all()

        if len(logins) > len(users):
            for user in users:
                logins.remove(user.login.lower())

            for login in logins:
                self.add_error('resp', 'Invalid login: {}'.format(login))

        return users

    def clean_trusted_sources(self):
        trusted_sources = self.cleaned_data['trusted_sources']
        if not trusted_sources:
            return set()

        names = set([source.strip() for source in trusted_sources.split(',')])
        sources = Source.query.filter(Source.name.in_(names)).all()
        existing_sources = names

        if len(sources) != len(names):
            existing_sources = {source.name for source in sources}
            for source in names - existing_sources:
                self.add_error('trusted_sources', 'Invalid source: {}'.format(source))

        return existing_sources

    def clean_flow(self):
        return self.cleaned_data['flow']

    def clean_key_sources(self):
        key_sources = self.cleaned_data['key_sources']
        if not key_sources:
            return set()

        names = {source.strip() for source in key_sources.split(',')}
        for name in names:
            if name not in KEY_SOURCE.choices():
                self.add_error('key_sources', 'Invalid key source: {}'.format(name))

        return names


class RemoveServerForm(forms.Form):
    srv = AlchemyFallbackChoiceField(
        queryset=lambda: Server.query,
        to_field_name='fqdn',
        error_messages={'invalid_choice': ERR_INVALID_SERVER},
    )
