import http.client
import json
import logging

import requests
from requests.adapters import HTTPAdapter
from requests.packages import urllib3

from django.conf import settings

logger = logging.getLogger(__name__)


class IdmApiError(Exception):
    pass


class IdmNetworkError(IdmApiError):
    pass


class IdmResponseError(IdmApiError):
    pass


class IdmClient(object):
    RETRY_METHOD_WHITELIST = {
        'GET', 'POST', 'PUT'
    }
    RETRY_STATUS_FORCELIST = {
        http.client.INTERNAL_SERVER_ERROR,
        http.client.BAD_GATEWAY,
    }

    def __init__(self):
        session = requests.Session()

        session.verify = settings.ROOT_CA_BUNDLE

        session.headers.update({
            'Authorization': 'OAuth {}'.format(settings.OAUTH_TOKEN),
            'Content-Type': 'application/json',
        })

        retries = urllib3.Retry(
            total=5,
            backoff_factor=0.1,
            method_whitelist=self.RETRY_METHOD_WHITELIST,
            status_forcelist=self.RETRY_STATUS_FORCELIST,
        )
        adapter = HTTPAdapter(max_retries=retries)
        session.mount('https://', adapter)

        self.session = session

    def perform_batch(self, batch_requests, dst_obj, message=''):

        def log_error(error_type, content=None):
            objs_info = '; '.join(
                '{} {} {}'.format(
                    request.get('method', ''),
                    request.get('path', '').split(settings.IDM_SYSTEM_NAME)[-1],
                    (request.get('body') or {}).get('name'),
                )
                for request in batch_requests
            )
            if content:
                try:
                    objs_info += ';\ncontent: {}'.format(content)
                except Exception as exc:
                    objs_info += ';\ncontent: {} decode error'.format(str(exc.__class__))
            logger.exception('Failed to reach idm with {}: operation {} dst {}  with requests:\n{}'.format(
                error_type,
                message,
                dst_obj,
                objs_info,
            ))

        try:
            response = self.session.post(
                url='{}/batch/'.format(settings.IDM_API_URL),
                data=json.dumps(batch_requests),
            )
        except Exception as error:
            err_type = 'timeout' if isinstance(error, requests.exceptions.ReadTimeout) else str(error.__class__)
            log_error(err_type)
            raise IdmNetworkError(error)

        logger.info('idm batch performed in {}'.format(response.elapsed))

        if response.status_code != 200:
            log_error('status code {}'.format(response.status_code), response.content[:10000])
            raise IdmResponseError(response)

    def fetch_rolenode_objects(self, slug_path):
        try:
            response = self.session.get(
                url='{}/rolenodes/'.format(settings.IDM_API_URL),
                timeout=settings.CAUTH_IDM_TIMEOUT,
                params={
                    'format': 'json',
                    'system': settings.IDM_SYSTEM_NAME,
                    'slug_path': slug_path,
                },
            )
        except Exception as error:
            logger.exception('Failed to reach idm')
            raise IdmNetworkError(error)

        logger.info('idm rolenodes fetched in {}'.format(response.elapsed))

        if response.status_code == 400:
            try:
                data = response.json()
            except ValueError:
                raise IdmResponseError(response)

            if 'errors' in data and 'slug_path' in data['errors']:
                return []

        if response.status_code != 200:
            logger.error(
                'Idm rolenode fetch failed: status code %s: %s',
                response.status_code, response.content
            )
            raise IdmResponseError(response)

        try:
            data = response.json()
        except ValueError:
            logger.exception('Failed to parse response json')
            raise IdmResponseError(response)

        return data['objects']
