import collections

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import (
    gr_m2m,
    sg_m2m,
    User,
    Server,
    ServerGroup,
)

from infra.cauth.server.master.notify.recipients import openstack_source_pattern, only_admins

SSH_ROLE_INFO = {
    'set': 'ssh',
    'name': {'en': 'Ssh', 'ru': 'Ssh'},
    'visibility': True,
    'fields': [
        {
            'required': False,
            'type': 'booleanfield',
            'slug': 'root',
            'name': {'en': 'root', 'ru': 'root'},
        },
    ],
}

SUDO_ROLE_INFO = {
    'set': 'sudo',
    'name': {'en': 'Sudo', 'ru': 'Sudo'},
    'visibility': True,
    'fields': [
        {
            'required': True,
            'type': 'choicefield',
            'slug': 'role',
            'name': {'en': 'role', 'ru': 'role'},
            'options': {
                'custom': True,
                'widget': 'textarea',
                'placeholder': 'where=(as_whom) commands',
                'validators': ['sudoers_entry'],
                'choices': [
                    {
                        'value': 'ALL=(ALL) ALL',
                        'name': {'en': 'ALL=(ALL) ALL', 'ru': 'ALL=(ALL) ALL'},
                    },
                    {
                        'value': 'ALL=(ALL) NOPASSWD: ALL',
                        'name': {'en': 'ALL=(ALL) NOPASSWD: ALL', 'ru': 'ALL=(ALL) NOPASSWD: ALL'},
                    },
                ],
            },
        },
    ],
}

EINE_ROLE_INFO = {
    'set': 'eine',
    'name': {'en': 'Access to eine', 'ru': 'Доступ в eine'},
    'visibility': True,
    'fields': [],
}


def responsible_json(login, notify):
    return {
        'username': login,
        'notify': notify,
    }


class BaseDst(object):
    def __init__(self, obj):
        self.obj = obj

    @property
    def key(self):
        raise NotImplementedError

    @property
    def request_email(self):
        raise NotImplementedError

    @property
    def notify_email(self):
        raise NotImplementedError

    @property
    def request_queue(self):
        raise NotImplementedError

    @property
    def type(self):
        raise NotImplementedError

    @property
    def responsibilities(self):
        raise NotImplementedError

    @property
    def dns_status(self):
        return None

    @property
    def sources(self):
        return set()

    def need_eine_node(self):
        """
        Говорит, нужен ли добавлять роль "доступ Eine" в idm.
        """
        return False

    @property
    def trusted_sources(self):
        return None

    def __hash__(self):
        return hash(self.key)

    def __eq__(self, other):
        if not isinstance(other, type(self)):
            return False

        return other.key == self.key

    @property
    def aliases(self):
        aliases = []
        type_names = (
            'type',
            'request_email',
            'notify_email',
            'request_queue',
            'dns_status',
            'trusted_sources',
        )

        for type_name in type_names:
            dst_type = getattr(self, type_name)
            if dst_type:
                aliases.append({
                    'type': type_name,
                    'name': {'en': dst_type, 'ru': dst_type},  # IDM оперирует локализованными данными
                })

        return aliases


class ServerDst(BaseDst):
    type = 'server'

    @property
    def key(self):
        return self.obj.fqdn

    @property
    def request_email(self):
        for group in self.obj.groups:
            if group.email is not None:
                return group.email

    @property
    def notify_email(self):
        for group in self.obj.groups:
            if group.notify_email is not None:
                return group.notify_email

    @property
    def request_queue(self):
        for group in self.obj.groups:
            if group.request_queue is not None:
                return group.request_queue

    @property
    def dns_status(self):
        if self.obj.flags is not None:
            return self.obj.flags.dns_status

    @property
    def sources(self):
        return {source.name for source in self.obj.sources}

    @property
    def aliases(self):
        aliases = super(ServerDst, self).aliases

        # Используется в воркфлоу системы Cauth в IDM
        if 'idm-cms' in self.trusted_sources and any((g.name.startswith('cms.') for g in self.obj.groups)):
            aliases.append({
                'type': 'need-idm-cms-approvers',
                'name': {'ru': 'true', 'en': 'true'},
            })
        walle_groups = [g.name for g in self.obj.groups if g.name.startswith('walle.')]
        if walle_groups:
            walle_project = walle_groups[0].split('.', 1)[1]
            aliases.append({
                'type': 'walle_project',
                'name': {'ru': walle_project, 'en': walle_project},
            })

        return aliases

    @property
    def trusted_sources(self):
        return ','.join(sorted([source.name for source in self.obj.trusted_sources]))

    def need_eine_node(self):
        return self.obj.is_baremetal

    @property
    def responsibilities(self):
        trusted_sources_ids = set(source.id for source in self.obj.trusted_sources)

        responsibility_sources = collections.defaultdict(bool)  # bool() == False
        openstack_admins = set()
        for resp in self.obj.responsibles:
            if trusted_sources_ids and resp.source_id not in trusted_sources_ids:
                continue

            is_openstack = openstack_source_pattern.match(resp.source.name)
            responsibility_sources[resp.user.login] |= not is_openstack
            if is_openstack:
                openstack_admins.add(resp.user)

        groups_responsibles = (
            Session.query(User.login)
            .select_from(Server)
            .join(sg_m2m, Server.id == sg_m2m.c.server_id)
            .join(ServerGroup, sg_m2m.c.group_id == ServerGroup.id)
            .join(gr_m2m, ServerGroup.id == gr_m2m.c.group_id)
            .join(User, gr_m2m.c.uid == User.uid)
            .filter(Server.id == self.obj.id)
            .distinct()
        )
        if trusted_sources_ids:
            groups_responsibles = (
                groups_responsibles.filter(ServerGroup.source_id.in_(trusted_sources_ids))
            )

        for (login,) in groups_responsibles:
            responsibility_sources[login] = True

        for user in only_admins(openstack_admins):
            responsibility_sources[user.login] = True

        return [responsible_json(login, notify) for login, notify in list(responsibility_sources.items())]


class GroupDst(BaseDst):
    type = 'group'

    @property
    def key(self):
        return self.obj.name

    @property
    def request_email(self):
        return self.obj.email

    @property
    def notify_email(self):
        return self.obj.notify_email

    @property
    def request_queue(self):
        return self.obj.request_queue

    @property
    def sources(self):
        return {self.obj.source.name}

    def need_eine_node(self):
        return 'bot' in self.sources

    @property
    def responsibilities(self):
        return [responsible_json(user.login, True) for user in self.obj.responsible_users]
