import socket
import tempfile
import subprocess

from django import forms

from infra.cauth.server.common.models import Role, User, Group, Server, ServerGroup

from infra.cauth.server.master.api.fields import AlchemyFallbackChoiceField

ERR_INVALID_USER = 'Person not found.'
ERR_INVALID_SRC = 'Person/group not found.'
ERR_INVALID_DST = 'Server/group not found.'
ERR_INVALID_ROLE = 'Role not found.'


def server_is_resolved(server):
    """ Don't check servers with default source in dns
    """
    if not any(source.is_default for source in server.sources):
        try:
            if not socket.getaddrinfo(server.fqdn, None):
                return False
        except Exception:
            return False

    return True


def is_correct_sudo_rule(sudo_rule):
    with tempfile.NamedTemporaryFile(mode='w') as f:
        f.write('login {}\n'.format(sudo_rule))
        f.flush()

        try:
            subprocess.check_call(('/usr/sbin/visudo', '-cf', f.name))
        except subprocess.CalledProcessError:
            return False

    return True


class RoleForm(forms.Form):
    role = forms.ChoiceField(choices=[
        ('ssh', 'ssh'),
        ('sudo', 'sudo'),
        ('eine', 'eine'),
    ])
    dst = forms.CharField()


class AddRuleForm(forms.Form):
    requester = AlchemyFallbackChoiceField(
        queryset=lambda: User.query,
        to_field_name='login',
        error_messages={'invalid_choice': ERR_INVALID_USER},
    )
    who = AlchemyFallbackChoiceField(
        queryset=[
            lambda: User.query.filter(User.is_fired.is_(False)),
            lambda: Group.query,
        ],
        to_field_name=['login', 'name'],
        error_messages={'invalid_choice': ERR_INVALID_SRC},
    )
    what = AlchemyFallbackChoiceField(
        queryset=[lambda: Server.query, lambda: ServerGroup.query],
        to_field_name=['fqdn', 'name'],
        error_messages={'invalid_choice': ERR_INVALID_DST},
    )
    description = forms.CharField()

    def clean_what(self):
        what = self.cleaned_data['what']

        if isinstance(what, Server) and not server_is_resolved(what):
            raise forms.ValidationError('Could not resolve ({})'.format(what.fqdn))

        return what


class AddSshForm(AddRuleForm):
    is_admin = forms.BooleanField(required=False)


class AddSudoForm(AddRuleForm):
    role = forms.CharField(max_length=1000, required=False)
    roleid = AlchemyFallbackChoiceField(
        queryset=lambda: Role.query,
        required=False,
        error_messages={'invalid_choice': ERR_INVALID_ROLE},
    )

    def clean_role(self):
        rolestr = self.cleaned_data['role'].strip()

        if not rolestr:
            return None

        if not is_correct_sudo_rule(rolestr):
            raise forms.ValidationError('Invalid sudo role')

        return rolestr

    def clean(self):
        data = super(AddSudoForm, self).clean()

        role = data.get('role')
        roleid = data.get('roleid')
        if (role and roleid) or (not role and not roleid):
            raise forms.ValidationError('Either role or roleid is required')

        return data


class AddEineForm(AddRuleForm):

    def clean_what(self):
        what = super(AddEineForm, self).clean_what()

        if isinstance(what, Server) and not what.is_baremetal:
            raise forms.ValidationError('Eine role is available only for hardware hosts')
        elif isinstance(what, ServerGroup) and what.source.name != 'bot':
            raise forms.ValidationError('Eine role is available only for bot groups')

        return what
