import logging
import uuid
import datetime
import urllib.request
import urllib.parse
import urllib.error

from django_idm_api.hooks import BaseHooks
from django_idm_api.exceptions import BadRequest, GroupNotFound

from django.conf import settings

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import Access, Server, Role, User, Group

from infra.cauth.server.master.api.views.base import FormError
from infra.cauth.server.master.api.idm.forms import RoleForm, AddSshForm, AddSudoForm, AddEineForm
from infra.cauth.server.master.api.idm.roles import SshRole, SudoRole, EineRole
from infra.cauth.server.master.api.idm.srcs import PersonSrc, GroupSrc
from infra.cauth.server.master.api.idm.streams import role_streams, EmptyPage, InvalidParams

logger = logging.getLogger(__name__)


class CauthIdmHooks(BaseHooks):
    def _add_role_for_src(self, src, role, fields):
        implementations = {
            'sudo': AddSudoHookImpl,
            'ssh': AddSshHookImpl,
            'eine': AddEineHookImpl,
        }

        role_form = RoleForm(role)
        if not role_form.is_valid():
            raise BadRequest(FormError(role_form).message)

        roledata = role_form.cleaned_data
        cls = implementations[roledata['role']]

        return cls().add_role_impl(src, roledata, fields)

    def add_role_impl(self, login, role, fields, **kwargs):
        return self._add_role_for_src(PersonSrc(login), role, fields)

    def add_group_role_impl(self, group, role, fields, **kwargs):
        return self._add_role_for_src(GroupSrc(group), role, fields)

    def _remove_role_for_src(self, src, role, data):
        implementations = {
            'sudo': RemoveSudoHookImpl,
            'ssh': RemoveAccessHookImpl,
            'eine': RemoveEineHookImpl,
        }

        role_form = RoleForm(role)
        if not role_form.is_valid():
            return

        roledata = role_form.cleaned_data
        cls = implementations[roledata['role']]
        cls().remove_role_impl(src, role, data)

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        return self._remove_role_for_src(PersonSrc(login), role, data)

    def remove_group_role_impl(self, group, role, data, is_deleted, **kwargs):
        return self._remove_role_for_src(GroupSrc(group), role, data)

    def get_roles(self, request):
        return RolesHookImpl().get_roles(request)


class HookImplBase(object):
    def __init__(self):
        self.request_id = str(uuid.uuid4()).split('-')[-1]

    def _log_request(self, action, params):
        params = ['{}={}'.format(k, v) for k, v in list(params.items())]
        logger.info(
            'new %s request id=%s: %s',
            action,
            self.request_id,
            ', '.join(params),
        )


class RolesHookImpl(HookImplBase):
    def get_roles(self, request):
        stream_id = request.GET.get('stream_id')
        if stream_id:
            try:
                stream = role_streams[stream_id]
            except KeyError:
                raise BadRequest('Invalid stream_id')
        else:
            stream = role_streams.get_first_stream()

        params = {k: v for k, v in list(request.GET.dict().items())
                  if k in stream.page_param_keys}

        while True:
            try:
                page = stream.get_page(**params)
                break
            except EmptyPage:
                params = {}
                stream = stream.next_stream
                if stream:
                    continue

                return {'code': 0, 'roles': []}
            except InvalidParams:
                raise BadRequest('Invalid page params')

        if page.is_last:
            next_stream = stream.next_stream
            next_page_params = {}
        else:
            next_stream = stream
            next_page_params = page.next_page_params

        result = {
            'code': 0,
            'roles': [role.render() for role in page.roles]
        }

        if next_stream:
            next_page_params['stream_id'] = next_stream.id
            result['next-url'] = '%s?%s' % (
                request.path, urllib.parse.urlencode(next_page_params))

        return result


class AddHookImplBase(HookImplBase):
    form_cls = None
    rule_type = None
    role_cls = None

    def add_role_impl(self, src, role, data):
        self._log_request('add-role', {
            'src': src,
            'role': role,
            'fields': data,
        })

        rule_input = self.get_form_input(src, role, data)
        form = self.form_cls(rule_input)

        if not form.is_valid():
            raise BadRequest(FormError(form).message)

        access_rule = self.get_existing_rule(form.cleaned_data)
        if access_rule:
            has_conflict = self.has_conflict(access_rule, form.cleaned_data)

            if self.is_rule_active(access_rule):
                if has_conflict:
                    raise BadRequest('Conflict with existing role detected.')
                logger.info(
                    'for request_id=%s: rule already exists #%s: %s',
                    self.request_id,
                    access_rule.id,
                    repr(access_rule),
                )
            else:
                self.resolve_conflict(access_rule, form.cleaned_data)
                self.activate_rule(access_rule)

                logger.info(
                    'for request_id=%s: updated old inactive rule #%s: %s',
                    self.request_id,
                    access_rule.id,
                    repr(access_rule),
                )
        else:
            access_rule = self.make_rule(form.cleaned_data)
            self.activate_rule(access_rule)
            Session.add(access_rule)
            Session.flush()

            logger.info(
                'for request_id=%s: added new rule #%s: %s',
                self.request_id,
                access_rule.id,
                repr(access_rule),
            )

        Session.commit()

        role = self.role_cls(src, access_rule)
        return {'data': role.fields}

    def get_existing_query(self, rule_data):
        return Access.query.filter_by(
            type=self.rule_type,
            src=str(rule_data['who']),
            dst=str(rule_data['what']),
        )

    def get_existing_rule(self, rule_data):
        return self.get_existing_query(rule_data).first()

    def get_form_input(self, src, role, data):
        if isinstance(src, PersonSrc):
            who = src.key
        elif isinstance(src, GroupSrc):
            group_obj = Group.query.filter_by(staff_id=src.key).first()
            if group_obj:
                who = group_obj.name
            else:
                raise GroupNotFound(
                    'group with staff_id={} not found'.format(src.key))
        else:
            raise TypeError

        return {
            'requester': settings.IDM_ROBOT_LOGIN,
            'who': who,
            'what': role['dst'],
            'description': 'added via idm',
        }

    def is_rule_active(self, rule):
        if rule.is_expired:
            return False

        if not (rule.approver and rule.approve_date):
            return False

        return True

    def activate_rule(self, rule):
        rule.requester = settings.IDM_ROBOT_LOGIN
        rule.approver = settings.IDM_ROBOT_LOGIN
        rule.request_date = datetime.datetime.now()
        rule.approve_date = datetime.datetime.now()
        rule.until = None

    def has_conflict(self, existing, rule_data):
        return False

    def resolve_conflict(self, existing, rule_data):
        pass

    def make_rule(self, rule_data):
        access = Access(
            type=self.rule_type,
            old_id=0,
            description=rule_data['description'],
            src=str(rule_data['who']),
            dst=str(rule_data['what']),
        )

        who = rule_data['who']
        if isinstance(who, User):
            access.src_user = who
        elif isinstance(who, Group):
            access.src_group = who

        if isinstance(rule_data['what'], Server):
            access.dst_server = rule_data['what']
        else:
            access.dst_group = rule_data['what']

        return access


class AddSshHookImpl(AddHookImplBase):
    form_cls = AddSshForm
    rule_type = 'ssh'
    role_cls = SshRole

    def get_form_input(self, src, role, data):
        data = data or {}
        super_self = super(AddSshHookImpl, self)
        super_data = super_self.get_form_input(src, role, data)
        return dict(super_data, is_admin=data.get('root', False))

    def make_rule(self, rule_data):
        access = super(AddSshHookImpl, self).make_rule(rule_data)
        if rule_data.get('is_admin', False):
            access.ssh_is_admin = True
        return access

    def has_conflict(self, existing, rule_data):
        return existing.ssh_is_admin != rule_data.get('is_admin', False)

    def resolve_conflict(self, existing, rule_data):
        existing.ssh_is_admin = rule_data.get('is_admin', False)


class AddSudoHookImpl(AddHookImplBase):
    form_cls = AddSudoForm
    rule_type = 'sudo'
    role_cls = SudoRole

    def get_form_input(self, src, role, data):
        data = data or {}
        super_self = super(AddSudoHookImpl, self)
        super_data = super_self.get_form_input(src, role, data)
        return dict(super_data, role=data.get('role', False))

    def make_rule(self, rule_data):
        role = Role.query.filter_by(spec=rule_data['role']).first()
        if not role:
            role = Role(spec=rule_data['role'])
            Session.add(role)

        access = super(AddSudoHookImpl, self).make_rule(rule_data)

        access.sudo_role = role
        return access

    def get_existing_query(self, rule_data):
        query = super(AddSudoHookImpl, self).get_existing_query(rule_data)
        return query.join('sudo_role').filter(Role.spec == rule_data['role'])


class AddEineHookImpl(AddHookImplBase):
    form_cls = AddEineForm
    rule_type = 'eine'
    role_cls = EineRole


class RemoveHookImplBase(HookImplBase):
    rule_type = None

    def remove_role_impl(self, src, role, data):
        self._log_request('remove-role', {
            'src': src,
            'role': role,
            'fields': data,
        })

        access = self.get_access_query(src, role, data).first()

        if access:
            logger.info(
                'for request_id=%s: deleting access #%s: %s',
                self.request_id,
                access.id,
                repr(access),
            )
            Session.delete(access)
            Session.commit()

    def get_access_query(self, src, role, data):
        query = Access.query.filter_by(
            type=self.rule_type,
            dst=role['dst'],
        )

        if isinstance(src, PersonSrc):
            query = query.filter_by(src=src.key)
        elif isinstance(src, GroupSrc):
            query = query.join('src_group').filter(Group.staff_id == src.key)
        else:
            raise TypeError

        return query


class RemoveAccessHookImpl(RemoveHookImplBase):
    rule_type = 'ssh'

    def get_access_query(self, src, role, data):
        data = data or {}
        super_self = super(RemoveAccessHookImpl, self)
        query = super_self.get_access_query(src, role, data)
        return query.filter(Access.ssh_is_admin.is_(data.get('root', False)))


class RemoveSudoHookImpl(RemoveHookImplBase):
    rule_type = 'sudo'

    def get_access_query(self, src, role, data):
        data = data or {}
        super_self = super(RemoveSudoHookImpl, self)
        query = super_self.get_access_query(src, role, data)
        return query.join('sudo_role').filter(Role.spec == data.get('role'))


class RemoveEineHookImpl(RemoveHookImplBase):
    rule_type = 'eine'
