import logging

from sqlalchemy.sql.functions import now

from django.conf import settings

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import Server, ServerGroup

from infra.cauth.server.master.api.idm.client import IdmClient
from infra.cauth.server.master.api.models import IdmUpdate
from infra.cauth.server.master.api.idm.dsts import (
    ServerDst,
    GroupDst,
    SSH_ROLE_INFO,
    SUDO_ROLE_INFO,
    EINE_ROLE_INFO,
)

logger = logging.getLogger(__name__)


def _get_api_node_key(item):
    item_name = item['name']
    name = item_name['en'] if isinstance(item_name, dict) else item_name
    return item['type'], name


class ApiNode(object):
    slug = None
    name = None
    set = None
    parent = None

    aliases = None
    fields = None
    visibility = None
    responsibilities = None

    def __init__(self, slug, name, parent, set_value=None, aliases=None, fields=None,
                 visibility=True, responsibilities=None):
        self.slug = slug
        self.parent = parent
        self.set = set_value

        if isinstance(name, dict):
            assert set(name) == {'en', 'ru'}
            self.name = name
        elif isinstance(name, str):
            self.name = {'en': name, 'ru': name}
        else:
            raise TypeError('Name should be either a string or a dict with en,ru strings')

        self.aliases = aliases or []
        self.fields = fields or []
        self.visibility = visibility
        self.responsibilities = responsibilities or []

    @property
    def slug_path(self):
        slugs = [_f for _f in self.parent.split('/') if _f] + [self.slug]
        return '/%s/' % '/'.join(slugs)

    @classmethod
    def from_api_object(cls, data):
        slugs = [_f for _f in data['slug_path'].split('/') if _f]
        if len(slugs) > 1:
            parent = '/%s/' % '/'.join(slugs[:-1])
        else:
            parent = None

        return cls(
            slug=data['slug'],
            name=data['name'],
            parent=parent,
            set_value=data.get('set'),
            aliases=data.get('aliases'),
            fields=data.get('fields'),
            visibility=data.get('visibility'),
            responsibilities=data.get('responsibilities'),
        )

    @classmethod
    def get_nodes(cls, dst):
        dst_node = cls(
            slug=dst.key,
            name=dst.key,
            parent='/dst/',
            aliases=dst.aliases,
            fields=None,
            responsibilities=dst.responsibilities,
        )

        role_key_node = cls(
            slug='role',
            name={
                'en': 'Role',
                'ru': 'Роль',
            },
            parent='/dst/%s/' % dst.key,
        )

        api_nodes = [dst_node, role_key_node]

        additional_roles = [SSH_ROLE_INFO, SUDO_ROLE_INFO]
        if dst.need_eine_node():
            additional_roles.append(EINE_ROLE_INFO)

        for role_info in additional_roles:
            api_nodes.append(
                ApiNode(
                    slug=role_info['set'],
                    set_value=role_info['set'],
                    name=role_info['name'],
                    parent='/dst/%s/role/' % dst.key,
                    aliases=None,
                    fields=role_info['fields'],
                    visibility=True,
                )
            )

        return api_nodes

    def as_tuple(self):
        flat_attrs = ('slug_path', 'name', 'set', 'visibility')
        items = [getattr(self, attr) for attr in flat_attrs]

        for attr in ('aliases', 'fields'):
            sorted_values = list(getattr(self, attr))
            sorted_values.sort(key=_get_api_node_key)
            items.append(sorted_values)

        sorted_responsibilities = sorted(self.responsibilities,
                                         key=lambda item: item['username'])
        items.append(sorted_responsibilities)

        return tuple(items)

    def get_create_data(self):
        data = {
            'system': settings.IDM_SYSTEM_NAME,
            'parent': self.parent,
            'slug': self.slug,
        }
        data.update(self.get_update_data())
        return data

    def get_update_data(self):
        return {
            'set': self.set,
            'name': self.name,
            'aliases': self.aliases or [],
            'fields': self.fields or [],
            'visibility': self.visibility,
            'responsibilities': self.responsibilities or [],
        }

    def __eq__(self, other):
        return (isinstance(other, type(self)) and
                self.as_tuple() == other.as_tuple())


class IdmUpdateAggregator(object):
    def __init__(self, suite_run_id):
        self.suite_run_id = suite_run_id
        self._destinations = set()

    def add_destination(self, dst):
        self._destinations.add(dst)

    def commit(self):
        connection = Session.connection()
        connection.execute('BEGIN')
        connection.execute('LOCK TABLE %s IN SHARE ROW EXCLUSIVE MODE' % IdmUpdate.__tablename__)

        # Создадим IdmUpdate, если их нет
        query = IdmUpdate.query.filter_by(suite_run_id=self.suite_run_id)
        existing_dsts = {row.dst for row in query}
        for dst in self._destinations - existing_dsts:
            delete_q = IdmUpdate.query.filter(
                IdmUpdate.suite_run_id != self.suite_run_id,
                IdmUpdate.dst == dst,
            )
            delete_q.delete(synchronize_session=False)

            upd = IdmUpdate(
                created_at=now(),
                dst=dst,
                suite_run_id=self.suite_run_id,
            )
            Session.add(upd)

        Session.flush()
        connection.execute('COMMIT')
        Session.commit()

    def finish_suite(self):
        from infra.cauth.server.master.api.tasks import push_idm_update

        query = IdmUpdate.query.filter_by(suite_run_id=self.suite_run_id)
        query.update({IdmUpdate.suite_is_finished: True}, synchronize_session=False)
        Session.commit()

        for update in query:
            push_idm_update.delay(update.dst)


def make_create_rolenode_request(api_node):
    return {
        'method': 'POST',
        'path': 'rolenodes/',
        'body': api_node.get_create_data(),
    }


def make_update_rolenode_request(api_node):
    path = 'rolenodes/%s%s' % (settings.IDM_SYSTEM_NAME, api_node.slug_path)
    return {
        'method': 'PUT',
        'path': path,
        'body': api_node.get_update_data(),
    }


def make_delete_rolenode_request(api_node):
    path = 'rolenodes/%s%s' % (settings.IDM_SYSTEM_NAME, api_node.slug_path)
    return {
        'method': 'DELETE',
        'path': path,
    }


def create_delete_dst_request(idm_client, dst_obj_name):
    remote_objects = idm_client.fetch_rolenode_objects('/dst/{}/'.format(dst_obj_name))
    remote_nodes = [ApiNode.from_api_object(obj) for obj in remote_objects]
    remote_node_map = {node.slug_path: node for node in remote_nodes}
    to_delete = sorted(remote_node_map.keys())
    if not to_delete:
        return []

    slug_path = to_delete[0]  # удалять нужно только корневой узел корневой узел
    logger.info('Local node does not exist: %s, deleting remote node', slug_path)

    remote_node = remote_node_map[slug_path]
    return [make_delete_rolenode_request(remote_node)]


def create_dst_requests(idm_client, dst_obj):
    if isinstance(dst_obj, Server):
        dst = ServerDst(dst_obj)
    elif isinstance(dst_obj, ServerGroup):
        dst = GroupDst(dst_obj)
    elif isinstance(dst_obj, str):
        return create_delete_dst_request(idm_client, dst_obj)
    else:
        raise TypeError(dst_obj)

    local_nodes = ApiNode.get_nodes(dst)
    local_node_map = {node.slug_path: node for node in local_nodes}

    remote_objects = idm_client.fetch_rolenode_objects('/dst/%s/' % dst.key)
    remote_nodes = list(map(ApiNode.from_api_object, remote_objects))
    remote_node_map = {node.slug_path: node for node in remote_nodes}

    to_add = sorted(set(local_node_map.keys()) - set(remote_node_map.keys()))  # создавать — сверху вниз
    to_maybe_update = sorted(set(remote_node_map.keys()) & set(local_node_map.keys()))

    requests = []

    for slug_path in to_add:
        local_node = local_node_map.get(slug_path)
        logger.info('Remote node does not exist: %s, creating remote node', slug_path)
        requests.append(make_create_rolenode_request(local_node))

    for slug_path in to_maybe_update:
        local_node = local_node_map.get(slug_path)
        remote_node = remote_node_map.get(slug_path)
        if local_node.as_tuple() != remote_node.as_tuple():
            logger.info(
                'Local and remote rolenodes differ: %s. local: %s. remote: %s',
                slug_path, local_node.as_tuple(), remote_node.as_tuple()
            )
            requests.append(make_update_rolenode_request(local_node))

    return requests


def ensure_remote_nodes_correct(dst_obj, message=''):
    """
    :param dst_obj: Server или ServerGroup
    :param message: сообщение с причиной пуша для записи в логи
    :return: False, если изменений нет, и узел уже был актуальным. True, если узлы были запушены
    """
    idm_client = IdmClient()
    requests = create_dst_requests(idm_client, dst_obj)
    if requests:
        idm_client.perform_batch(requests, dst_obj, message)
        return True
    else:
        logger.info('No updates required')
        return False
