import datetime

from infra.cauth.server.common.models import BaseModel, Server, ServerGroup
from sqlalchemy.orm import configure_mappers, relationship, backref
from sqlalchemy import (
    Column,
    String,
    DateTime,
    Boolean,
    Enum,
    PrimaryKeyConstraint,
    ForeignKey,
    Text,
    Integer,
)


class ConfigValue(BaseModel):
    __tablename__ = 'config_values'

    id = Column(String(32), primary_key=True)
    value = Column(String(255))


class IdmUpdate(BaseModel):
    __tablename__ = 'idm_updates'
    __table_args__ = (PrimaryKeyConstraint('dst', 'suite_run_id'), )

    created_at = Column(DateTime)
    started_at = Column(DateTime, nullable=True)
    dst = Column(String(128))
    suite_run_id = Column(String(36))
    suite_is_finished = Column(Boolean)  # don't apply update until this is true

    def get_object(self):
        server_query = Server.query.filter_by(fqdn=self.dst)
        group_query = ServerGroup.query.filter_by(name=self.dst)
        return server_query.first() or group_query.first() or self.dst


class EnabledWikiGroup(BaseModel):
    __tablename__ = 'enabled_wiki_groups'

    id = Column(Integer, primary_key=True)
    created_at = Column(DateTime)
    name = Column(String(128), unique=True)
    description = Column(Text)


class DNS_STATUS(object):
    OK = 'ok'
    SOME_PTR_MISSING = 'some_ptr_missing'
    CNAME = 'cname'
    HAS_NO_PTRS = 'has_no_ptrs'
    CANT_RESOLVE = 'cant_resolve'

    status_names = [
        OK,
        SOME_PTR_MISSING,
        CNAME,
        HAS_NO_PTRS,
        CANT_RESOLVE,
    ]

    bad_statuses = [
        HAS_NO_PTRS,
        CANT_RESOLVE,
    ]


class ServerFlags(BaseModel):
    __tablename__ = 'server_flags'

    server_id = Column(Integer, ForeignKey(Server.id, ondelete='CASCADE'), primary_key=True)
    dns_status = Column(Enum(*DNS_STATUS.status_names, name='dns_status'))
    dns_status_expire_at = Column(DateTime)

    server = relationship(
        Server,
        uselist=False,
        backref=backref('flags', uselist=False, passive_deletes=True),
    )

    @property
    def is_expired(self):
        return self.dns_status_expire_at < datetime.datetime.now()


configure_mappers()
