import json
import time
import logging
import collections
import datetime

from django.conf import settings
from django.views.generic import View
from django.http import HttpResponse, HttpResponseForbidden
from django_tools_log_context import request_log_context
from django.utils.functional import cached_property

from infra.cauth.server.master.utils.json_encoder import CustomEncoder
from infra.cauth.server.master.utils.config import service_is_readonly

logger = logging.getLogger(__name__)


class RequestError(ValueError):
    @property
    def message(self):
        return self.args[0]


class FormError(RequestError):
    def __init__(self, form):
        self.form = form

    @property
    def message(self):
        field_errors = []
        for key, values in list(self.form.errors.items()):
            values = '; '.join(values)
            if key == '__all__':
                field_errors.append(values)
            else:
                field_errors.append('%s: %s' % (key, values))
        return '; '.join(field_errors)


class ContextMixin(object):
    def dispatch(self, request, *args, **kwargs):
        with request_log_context(request, endpoint=self, threshold=0):
            super(ContextMixin, self).dispatch(request, *args, **kwargs)


class BaseView(View):
    REQUIRE_CERT = None
    IS_READ_ONLY = False
    PRETTY_JSON = True

    def __init__(self):
        super(BaseView, self).__init__()
        self.golem_req_params = {}

    def dispatch(self, request, *args, **kwargs):
        try:
            return self.inner_dispatch(request, *args, **kwargs)
        finally:
            self.log_golem_req()

    def log_golem_req(self):
        request = self.request
        params = collections.defaultdict(str)
        params.update(self.params)

        data = collections.OrderedDict([
            ('url', request.path),
            ('golem_ip', request.META['REMOTE_ADDR']),
            ('status', ''),
            ('user', params['requester'] or params['login']),
            ('src_ip', params['ip']),
            ('token', params['token']),
            ('who', params['who']),
            ('what', params['what']),
            ('role', params['role']),
            ('id', params['id']),
            ('approvers', ''),
            ('srv', params['srv']),
            ('grp', params['grp']),
            ('resp', params['resp']),
        ])

        data.update(self.golem_req_params)

        for key in ('approvers', 'grp', 'resp'):
            if isinstance(data[key], (list, set)):
                data[key] = ','.join(data[key])

        msg = ' '.join('{0}="{1}"'.format(*item) for item in list(data.items()))
        ts = datetime.datetime.now().strftime(
            '%a %b %d %H:%M:%S ' + time.tzname[0] or ''
        )
        logger.info('%s: %s', ts, msg)

    def inner_dispatch(self, request, *args, **kwargs):
        if self.REQUIRE_CERT:
            verified = request.META.get('HTTP_X_SSL_CLIENT_VERIFY')
            if verified != '0':
                return HttpResponseForbidden('Not authorized')
            sources = settings.CERT_ACCESS_REGISTRY[self.REQUIRE_CERT]
            request.source_name = None
            for source_dict in sources:
                if request.META.get('HTTP_X_SSL_CLIENT_SUBJECT') in source_dict['certs']:
                    request.source_name = source_dict['source']
                    request.should_set_source = source_dict.get('set_source', False)
                    break

            if request.source_name is None:
                self.golem_req_params['status'] = 'forbidden'
                message = 'Your certificate ({}) is not authorized to use this CAuth method'.format(
                    request.META.get('HTTP_X_SSL_CLIENT_SUBJECT')
                )
                return HttpResponseForbidden(
                    content=message,
                    content_type='text/plain; charset=utf-8',
                )

        try:
            if not self.IS_READ_ONLY and service_is_readonly():
                self.golem_req_params['status'] = 'readonly'
                raise RequestError('service is in readonly state')
            else:
                result = super(BaseView, self).dispatch(request, *args, **kwargs)
        except RequestError as error:
            if 'status' not in self.golem_req_params:
                self.golem_req_params['status'] = 'bad request'
            return HttpResponse(
                error.message,
                content_type='text/plain; charset=utf-8',
            )

        if isinstance(result, HttpResponse):
            return result
        elif isinstance(result, str):
            content_type = 'text/plain; charset=utf-8'
            data = result
        else:
            if (isinstance(result, dict) and 'status' in result
                    and 'status' not in self.golem_req_params):
                self.golem_req_params['status'] = result['status']

            content_type = 'application/json; charset=utf-8'
            data = json.dumps(result, cls=CustomEncoder,
                              indent=4 if self.PRETTY_JSON else None,
                              ensure_ascii=False)

        return HttpResponse(data, content_type=content_type)

    @cached_property
    def params(self):
        if self.request.method.lower() == 'get':
            return self.request.GET.dict()
        elif self.request.method.lower() == 'post':
            data = self.request.POST.dict()
            if 'req' in data:
                try:
                    data = json.loads(data['req'])
                except ValueError:
                    raise RequestError('req is not a valid json')

                if isinstance(data, dict):
                    return data
                else:
                    raise RequestError('req is not a dict')
            else:
                return data
        else:
            return {}

    @staticmethod
    def check_authoritative_permission(server, source):
        authoritative_source_id = getattr(server.authoritative_group, 'source_id', None)
        if authoritative_source_id is None or authoritative_source_id == source.id:
            return True
        return False
