import json
import logging

from sqlalchemy import orm

from infra.cauth.server.common.models import Server, ServerGroup, ServerResponsible

from infra.cauth.server.master.cache.utils import SafetyWriteFile
from infra.cauth.server.master.constants import FILE_TYPE
from infra.cauth.server.master.utils.tasks import task
from infra.cauth.server.master.api.idm.dsts import ServerDst, GroupDst, SSH_ROLE_INFO, SUDO_ROLE_INFO

logger = logging.getLogger(__name__)


@task(dedicated_logger=False)
def update_idm_info_cache():
    """Генерирует дерево ролей и кеширует в файл settings.IDM_INFO_FILE
    В последствии отдается по запросу /idm/info/ средствами nginx
    Не вызывается из кода. Должна быть запущена руками перед синхронизацией дерева IDM
    """
    servers_query = (
        Server.query
        .options(
            orm.joinedload(Server.flags),
            orm.joinedload(Server.trusted_sources),
            orm.subqueryload(Server.groups),
            orm.subqueryload(Server.responsibles).joinedload(ServerResponsible.user),
            orm.subqueryload(Server.responsibles).joinedload(ServerResponsible.source),
        )
    )

    groups_query = (
        ServerGroup.query
        .options(
            orm.subqueryload(ServerGroup.servers).joinedload(Server.flags),
            orm.subqueryload(ServerGroup.responsible_users),
        )
    )

    dst_queries = [
        (ServerDst, servers_query),
        (GroupDst, groups_query),
    ]

    with IdmCacheFile() as idm_info_file:
        for dst_class, query in dst_queries:
            for obj in query:
                dst = dst_class(obj)
                idm_info_file.write(dst)


class IdmCacheFile(SafetyWriteFile):
    FILE_TYPE = FILE_TYPE.IDM_CACHE
    filler_separator = '"a":0'
    filler = {
        'code': 0,
        'roles': {
            'slug': 'dst',
            'name': {
                'ru': 'Назначение',
                'en': 'Destination',
            },
            'values': {'a': 0},
        },
    }

    def write(self, dst):
        dst_object = render_info(dst)

        # [1:-1] убирает фигурные скобки по краям объекта
        dst_json = json.dumps(dst_object, separators=(',', ':'))[1:-1]

        super(IdmCacheFile, self).write(dst_json)


def render_info(dst):
    return {
        dst.key: {
            'name': dst.key,
            'aliases': dst.aliases,
            'responsibilities': dst.responsibilities,
            'visibility': True,
            'roles': {
                'slug': 'role',
                'name': {
                    'ru': 'Роль',
                    'en': 'Role',
                },
                'values': {
                    'ssh': SSH_ROLE_INFO,
                    'sudo': SUDO_ROLE_INFO,
                },
            },
        },
    }
