import abc
import json
import functools
import itertools

import attr

from django.conf import settings
from sqlalchemy import and_, exists, or_

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.constants import FLOW_TYPE
from infra.cauth.server.common.models import (
    Access,
    User,
    Server,
    Group,
    Source,
    ServerGroup,
    ServerGroupTrustedSourceRelation,
    ServerResponsible,
    ServerTrustedSourceRelation,
    sg_m2m,
)

from infra.cauth.server.master.cache.utils import SafetyWriteFile
from infra.cauth.server.master.constants import FILE_TYPE
from infra.cauth.server.master.utils.tasks import task

RESULT_CHUNK_SIZE = 10000


class PuncherCacheFile(SafetyWriteFile):
    FILE_TYPE = FILE_TYPE.PUNCHER_CACHE
    filler_separator = '"a"'
    filler = {"rules": ['a']}

    def write(self, rule):
        rule_object = rule.get_puncher_object()
        if rule_object is not None:  # TODO: revert later, when puncher will support YP
            rule_json = json.dumps(rule_object, separators=(',', ':'))
        else:
            rule_json = None

        super(PuncherCacheFile, self).write(rule_json)


class PuncherBaseRule(object, metaclass=abc.ABCMeta):
    @abc.abstractmethod
    def get_puncher_object(self):
        return {}


@attr.s(slots=True, hash=True)
class PuncherSrcUserRule(PuncherBaseRule):
    uid = attr.ib()
    login = attr.ib()

    def get_puncher_object(self):
        return {
            'type': 'user',
            'object': {
                'uid': self.uid,
                'login': self.login,
            }
        }


@attr.s(slots=True, hash=True)
class PuncherSrcGroupRule(PuncherBaseRule):
    type = attr.ib()
    gid = attr.ib()
    name = attr.ib()
    staff_id = attr.ib()
    service_id = attr.ib()

    def get_puncher_object(self):
        result = {
            'type': self.type,
            'object': {
                'gid': self.gid,
                'name': self.name,
                'staff_id': self.staff_id,
            }
        }

        if self.type in ('svc', 'svcrole'):
            result['object']['service_id'] = self.service_id

        return result


@attr.s(slots=True, hash=True)
class PuncherDstServerRule(PuncherBaseRule):
    id = attr.ib()
    fqdn = attr.ib()
    type = attr.ib()

    def get_puncher_object(self):
        return {
            'type': self.type,
            'object': {
                'id': self.id,
                'fqdn': self.fqdn,
            }
        }


@attr.s(slots=True, init=False, hash=True)
class PuncherDstGroupRule(PuncherBaseRule):
    id = attr.ib()
    source = attr.ib()
    name = attr.ib()

    def __init__(self, group_id, name):
        self.id = group_id
        self.source, self.name = name.split('.', 1)

    def get_puncher_object(self):
        if self.source not in settings.PUNCHER_SOURCES_WHITE_LIST:  # TODO: revert when pucnher will support them
            return None
        return {
            'type': 'group',
            'object': {
                'id': self.id,
                'name': self.name,
                'source': self.source,
            }
        }


@attr.s(slots=True, hash=True)
class PuncherRule(PuncherBaseRule):
    src = attr.ib()
    dst = attr.ib()
    is_responsible = attr.ib(cmp=False)

    def get_puncher_object(self):
        result = {
            'src': self.src.get_puncher_object(),
            'dst': self.dst.get_puncher_object(),
        }
        if result['dst'] is None:  # TODO: revert when pucnher will support YP
            return None

        if self.is_responsible:
            result['is_responsible'] = self.is_responsible

        return result


def get_user_server_access_rules(base_query, puncher_file):
    query = (
        base_query
        .filter(Access.src_user.has(), Access.dst_server.has())
        .with_entities(
            User.uid, User.login,
            Access.dst_server_id, Access.dst, Server.type,
        )
    )

    for uid, login, server_id, fqdn, server_type in query:
        rule = PuncherRule(
            src=PuncherSrcUserRule(uid, login),
            dst=PuncherDstServerRule(server_id, fqdn, server_type),
            is_responsible=False,
        )
        puncher_file.write(rule)


def get_user_group_access_rules(base_query, puncher_sources_pks, puncher_file):
    query = (
        base_query
        .filter(
            Access.src_user.has(),
            ServerGroup.source_id.in_(puncher_sources_pks)
        )
        .with_entities(
            User.uid, User.login,
            ServerGroup.id, ServerGroup.name,
        )
    )

    for uid, login, group_id, name in query:
        rule = PuncherRule(
            src=PuncherSrcUserRule(uid, login),
            dst=PuncherDstGroupRule(group_id, name),
            is_responsible=False,
        )
        puncher_file.write(rule)


def get_user_group_expandable_access_rules(base_query, puncher_sources_pks, puncher_file):
    query = (
        base_query
        .filter(
            Access.src_user.has(),
            ~ServerGroup.source_id.in_(puncher_sources_pks)
        )
        .with_entities(
            User.uid, User.login,
            Server.id, Server.fqdn, Server.type,
        )
    )

    for uid, login, server_id, name, server_type in query:
        rule = PuncherRule(
            src=PuncherSrcUserRule(uid, login),
            dst=PuncherDstServerRule(server_id, name, server_type),
            is_responsible=False,
        )
        puncher_file.write(rule)


def get_group_server_access_rules(base_query, puncher_file):
    query = (
        base_query
        .join(Group)
        .filter(Access.src_group.has(), Access.dst_server.has())
        .with_entities(
            Group.type, Group.gid, Group.name, Group.staff_id, Group.service_id,
            Access.dst_server_id, Access.dst, Server.type,
        )
    )

    for group_type, gid, name, staff_id, service_id, server_id, fqdn, server_type in query:
        rule = PuncherRule(
            src=PuncherSrcGroupRule(group_type, gid, name, staff_id, service_id),
            dst=PuncherDstServerRule(server_id, fqdn, server_type),
            is_responsible=False,
        )
        puncher_file.write(rule)


def get_group_group_access_rules(base_query, puncher_sources_pks, puncher_file):
    query = (
        base_query
        .join(Group)
        .filter(
            Access.src_group.has(),
            ServerGroup.source_id.in_(puncher_sources_pks)
        )
        .with_entities(
            Group.type, Group.gid, Group.name, Group.staff_id, Group.service_id,
            ServerGroup.id, ServerGroup.name,
        )
    )

    for group_type, gid, src_name, staff_id, service_id, server_group_id, dst_name in query:
        rule = PuncherRule(
            src=PuncherSrcGroupRule(group_type, gid, src_name, staff_id, service_id),
            dst=PuncherDstGroupRule(server_group_id, dst_name),
            is_responsible=False,
        )
        puncher_file.write(rule)


def get_group_group_expandable_access_rules(base_query, puncher_sources_pks, puncher_file):
    query = (
        base_query
        .join(Group)
        .filter(
            Access.src_group.has(),
            ~ServerGroup.source_id.in_(puncher_sources_pks)
        )
        .with_entities(
            Group.type, Group.gid, Group.name, Group.staff_id, Group.service_id,
            Server.id, Server.fqdn, Server.type,
        )
    )

    for group_type, gid, src_name, staff_id, service_id, server_id, fqdn, server_type in query:
        rule = PuncherRule(
            src=PuncherSrcGroupRule(group_type, gid, src_name, staff_id, service_id),
            dst=PuncherDstServerRule(server_id, fqdn, server_type),
            is_responsible=False,
        )
        puncher_file.write(rule)


def get_server_responsible_rules(puncher_file):
    """ Эта часть генерирует больше всего дублей из-за того, что игнорируются sources
    По хорошему нужно решать с помошью гарантирования уникальности в ServerResponsible
    (вынести связь с source в отдельную модель). Так же здесь потребляется основное количество
    памяти, нужно организовать постраничный обход (даст общее замедление ~1мин)
    """
    backend_sources_servers = (
        Session.query(
            Server.id
        ).join(
            sg_m2m, ServerGroup
        ).join(
            Source, ServerGroup.source_id == Source.id
        ).filter(
            ServerGroup.flow == FLOW_TYPE.BACKEND_SOURCES,
            Source.is_modern.is_(True),
        ).subquery()
    )

    resps_for_classic_servers_without_trusted = (
        Session.query(ServerResponsible).yield_per(
            RESULT_CHUNK_SIZE
        ).join(User, Server).filter(
            ~Server.classic_trusted_sources.any(),
            ~exists().where(Server.id == backend_sources_servers.c.id),
        ).with_entities(User.uid, User.login, Server.id, Server.fqdn, Server.type)
    )

    resps_for_classic_servers_with_trusted = (
        Session.query(ServerResponsible).yield_per(
            RESULT_CHUNK_SIZE
        ).join(
            User, Server, ServerTrustedSourceRelation
        ).filter(
            ServerResponsible.source_id == ServerTrustedSourceRelation.source_id,
            ~exists().where(Server.id == backend_sources_servers.c.id),
        ).with_entities(User.uid, User.login, Server.id, Server.fqdn, Server.type)
    )

    resps_for_backend_sources_servers = Session.query(ServerResponsible).yield_per(
        RESULT_CHUNK_SIZE
    ).join(
        User, Server, sg_m2m
    ).join(
        ServerGroup, and_(
            ServerGroup.flow == FLOW_TYPE.BACKEND_SOURCES,
            sg_m2m.c.group_id == ServerGroup.id,
        )
    ).join(
        Source, and_(
            Source.id == ServerGroup.source_id,
            Source.is_modern.is_(True),
        )
    ).join(
        ServerGroupTrustedSourceRelation,
        ServerGroupTrustedSourceRelation.servergroup_id == ServerGroup.id,
    ).filter(
        ServerResponsible.source_id == ServerGroupTrustedSourceRelation.source_id,
    ).with_entities(User.uid, User.login, Server.id, Server.fqdn, Server.type)

    query = itertools.chain(
        resps_for_classic_servers_without_trusted,
        resps_for_classic_servers_with_trusted,
        resps_for_backend_sources_servers,
    )

    for uid, login, server_id, fqdn, server_type in query:
        rule = PuncherRule(
            src=PuncherSrcUserRule(uid, login),
            dst=PuncherDstServerRule(server_id, fqdn, server_type),
            is_responsible=True,
        )
        puncher_file.write(rule)


def get_server_group_responsible_rules(puncher_sources_pks, puncher_file):
    query = (
        Session.query(ServerGroup.id, ServerGroup.name, User.uid, User.login)
        .yield_per(RESULT_CHUNK_SIZE)
        .join(ServerGroup.source)
        .join(ServerGroup.responsible_users)
        .filter(ServerGroup.source_id.in_(puncher_sources_pks))
    )

    for g_id, g_name, u_uid, u_login in query:
        rule = PuncherRule(
            src=PuncherSrcUserRule(u_uid, u_login),
            dst=PuncherDstGroupRule(g_id, g_name),
            is_responsible=True,
        )
        puncher_file.write(rule)


def get_server_group_responsible_expandable_rules(puncher_sources_pks, puncher_file):
    query = (
        Session.query(ServerGroup)
        .yield_per(RESULT_CHUNK_SIZE)
        .join(ServerGroup.responsible_users)
        .join(ServerGroup.servers)
        .filter(~ServerGroup.source_id.in_(puncher_sources_pks))
        .with_entities(User.uid, User.login, Server.id, Server.fqdn, Server.type)
        .distinct()
    )

    for user_uid, login, server_id, server_fqdn, server_type in query:
        rule = PuncherRule(
            src=PuncherSrcUserRule(user_uid, login),
            dst=PuncherDstServerRule(server_id, server_fqdn, server_type),
            is_responsible=True,
        )
        puncher_file.write(rule)


@task(dedicated_logger=False)
def update_puncher_rules_cache():
    """Генерирует ответ для puncher в settings.PUNCHER_RULES_FILE
    В последствии отдается по запросу /puncher/rules/ средствами nginx
    """

    base_query = (
        Session.query(Access)
        .yield_per(RESULT_CHUNK_SIZE)
        .outerjoin(Access.src_user)
        .filter(
            Access.type == 'ssh',
            or_(
                User.uid.is_(None),
                User.is_fired.is_(False),
            )
        )
    )

    # TODO(lavrukov): как-нибудь отрефакторить это после мерджа репозиториев
    # Это Access.get_active_query() использованное с сессией
    base_query = Access.add_active_filter(base_query)

    base_server_query = (
        base_query
        .join(Access.dst_server)
    )

    base_group_query = (
        base_query
        .join(Access.dst_group)
        .join(ServerGroup.servers)
        .distinct()
    )

    puncher_sources_pks = Session.query(Source.id).filter(
        Source.name.in_(settings.PUNCHER_SOURCES_WHITE_LIST)
    ).all()

    # NOTE: Здесь появляется некоторое количество дублей за счет того, что некоторые люди из access
    # правил так же входят в список ответсвенных
    getters = [
        get_server_responsible_rules,
        functools.partial(get_server_group_responsible_rules, puncher_sources_pks),
        functools.partial(get_server_group_responsible_expandable_rules, puncher_sources_pks),
        functools.partial(get_user_server_access_rules, base_server_query),
        functools.partial(get_user_group_access_rules, base_group_query, puncher_sources_pks),
        functools.partial(
            get_user_group_expandable_access_rules, base_group_query, puncher_sources_pks
        ),
        functools.partial(get_group_server_access_rules, base_server_query),
        functools.partial(get_group_group_access_rules, base_group_query, puncher_sources_pks),
        functools.partial(
            get_group_group_expandable_access_rules, base_group_query, puncher_sources_pks
        ),
    ]

    with PuncherCacheFile() as puncher_file:
        for getter in getters:
            getter(puncher_file)
