import os
import json
import tempfile

from django.conf import settings

from infra.cauth.server.master.constants import FILE_TYPE
from infra.cauth.server.master.files.models import S3File


class SafetyWriteFile(object):
    """ Обертка для последовательной записи в файл обьектов, разделенных запятой. После окончания
    записи происходит загрузка временного файла, в который велась запись, в конечный файл S3File.
    filler - json в который будут вставлятся обьекты. В месте, в котором мы ожидаем их увидеть
    нужно вставить особый обькт, который будет уникальным (как подстрока) в сериализованном filler.
    Этот сериализованный объект необходимо указать в filler_separator. На его место будут
    записываться объекты.
    """
    FILE_TYPE = None
    filler_separator = None
    filler = None

    def __init__(self):
        self.is_first_write = True

        filler = json.dumps(self.filler, separators=(',', ':'))
        fillers = filler.split(self.filler_separator)
        self.front_filler = fillers[0]
        self.trailing_filler = fillers[1]

        self.tmp_file = tempfile.NamedTemporaryFile(mode='w', dir=settings.TMP_DIR, delete=False)

    def __enter__(self):
        self.tmp_file.__enter__()

        self.tmp_file.write(self.front_filler)

        return self

    def write(self, obj):
        if obj is None:
            # TODO: revert when pucnher will support YP
            return
        if not self.is_first_write:
            self.tmp_file.write(',')

        self.tmp_file.write(obj)

        self.is_first_write = False

    def __exit__(self, exc, value, tb):
        if exc is not None:
            result = self.tmp_file.__exit__(exc, value, tb)
            os.unlink(self.tmp_file.name)
            return result

        self.tmp_file.write(self.trailing_filler)

        result = self.tmp_file.__exit__(exc, value, tb)

        file_type = self.FILE_TYPE or FILE_TYPE.NOT_SET

        try:
            with open(self.tmp_file.name, 'rb') as tmp_fileobj:
                S3File.create_obj(file_type, file_type, tmp_fileobj.read())
        finally:
            os.remove(self.tmp_file.name)

        return result
