import re
import json
import pprint

from marshmallow import ValidationError

from django import forms
from django.forms.utils import ErrorList
from django.utils.safestring import mark_safe

from django_celery_beat.models import PeriodicTask, IntervalSchedule, CrontabSchedule, \
    PERIOD_CHOICES

from infra.cauth.server.common.models import User, Group

from infra.cauth.server.master.importers.registry import SuiteGroup
from infra.cauth.server.master.celery_app import app
from infra.cauth.server.master.api.fields import AlchemyFallbackChoiceField
from infra.cauth.server.master.api.views.base import FormError


class BootstrapErrorList(ErrorList):
    def as_ul(self):
        html = super(BootstrapErrorList, self).as_ul()
        if html:
            html = '<div class="alert alert-danger">{}</div>'.format(html)

        return mark_safe(html)


class ImportsDeleteForm(forms.Form):
    group = forms.ChoiceField(
        choices=[(key, key) for key in SuiteGroup.registry]
    )

    def clean_group(self):
        return SuiteGroup.registry[self.cleaned_data['group']]


class ImportsDownloadForm(ImportsDeleteForm):
    type = forms.ChoiceField(choices=(
        ('original', 'original'),
        ('override', 'override'),
    ))


class ImportUploadForm(ImportsDeleteForm):
    override = forms.FileField()

    def clean_override(self):
        try:
            return json.load(self.cleaned_data['override'])
        except ValueError:
            raise forms.ValidationError('Invalid json')

    def clean(self):
        group = self.cleaned_data.get('group')
        if not group:
            return self.cleaned_data

        if self.errors:
            return self.cleaned_data

        try:
            group.validator().validate(self.cleaned_data['override'])
        except ValidationError as error:
            error = error.messages
            error = pprint.pformat(error)
            self.add_error('override', error)

        return self.cleaned_data


class ImportUnlockForm(ImportsDeleteForm):
    suite = forms.CharField()

    def clean(self):
        suite = self.cleaned_data.get('suite')
        group = self.cleaned_data.get('group')

        if not (suite or group):
            return self.cleaned_data

        if suite in group.suites:
            self.cleaned_data['suite'] = group.suites[suite]
        else:
            self.add_error('suite', 'Invalid suite for this group')

        return self.cleaned_data


class PeriodicTaskForm(forms.ModelForm):
    class Meta(object):
        model = PeriodicTask
        fields = ['name', 'task', 'args', 'kwargs', 'enabled']
        widgets = {
            'args': forms.TextInput(),
            'kwargs': forms.TextInput(),
        }

    schedule = forms.CharField(max_length=64)

    interval = forms.CharField(required=False)
    crontab = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        super(PeriodicTaskForm, self).__init__(*args, **kwargs)
        task_choices = [(name, name) for name, task in list(app.tasks.items())
                        if hasattr(task, 'info')]
        task_choices.insert(0, ('', ''))
        self.fields['task'] = forms.ChoiceField(choices=task_choices,
                                                initial='')
        self.error_class = BootstrapErrorList

    def clean_task(self):
        task_path = self.cleaned_data['task']
        if not task_path:
            raise forms.ValidationError("Field is required.")

        return task_path

    def clean_args(self):
        args = self.cleaned_data['args']
        try:
            value = json.loads(args)
        except Exception:
            raise forms.ValidationError("Not a valid json.")

        if not isinstance(value, list):
            raise forms.ValidationError("Must be a json list.")

        return args

    def clean_kwargs(self):
        kwargs = self.cleaned_data['kwargs']
        try:
            value = json.loads(kwargs)
        except Exception:
            raise forms.ValidationError("Not a valid json.")

        if not isinstance(value, dict):
            raise forms.ValidationError("Must be a json dict.")

        return kwargs

    @staticmethod
    def _detect_interval(components):
        if len(components) != 2:
            return None

        digit = re.compile(r'^\d+$')
        if not digit.match(components[0]):
            return None

        period_ids = [p[0] for p in PERIOD_CHOICES]
        test_plural = components[1]
        test_pluralized = components[1] + 's'
        if test_plural in period_ids:
            period = test_plural
        elif test_pluralized in period_ids:
            period = test_pluralized
        else:
            return None

        every = int(components[0])

        try:
            return IntervalSchedule.objects.get(every=every, period=period)
        except IntervalSchedule.DoesNotExist:
            return IntervalSchedule.objects.create(every=every, period=period)

    @staticmethod
    def _detect_crontab(components):
        if len(components) != 5:
            return None

        fields = ['minute', 'hour', 'day_of_week', 'day_of_month',
                  'month_of_year']
        attrs = dict(list(zip(fields, components)))

        try:
            return CrontabSchedule.objects.get(**attrs)
        except CrontabSchedule.DoesNotExist:
            return CrontabSchedule.objects.create(**attrs)

    def clean_schedule(self):
        schedule_str = self.cleaned_data['schedule']

        test = re.compile(r'\s{2,}')
        replace = re.compile(r'\s+')

        while test.search(schedule_str):
            schedule_str = re.sub(replace, ' ', schedule_str)

        components = schedule_str.split(' ')
        interval = self._detect_interval(components)
        if interval:
            self.instance.interval = interval
            self.instance.crontab = None
            return interval

        crontab = self._detect_crontab(components)
        if crontab:
            self.instance.crontab = crontab
            self.instance.interval = None
            return crontab

        raise forms.ValidationError("Unknown schedule")


class AccessDeltaForm(forms.Form):
    obj = AlchemyFallbackChoiceField(
        queryset=[
            lambda: User.query.filter(User.is_fired.is_(False)),
            lambda: Group.query.filter_by(type='dpt'),
        ],
        to_field_name=['login', 'name'],
        error_messages={'invalid_choice': 'Invalid object'},
        widget=forms.TextInput(),
    )

    dst = AlchemyFallbackChoiceField(
        queryset=lambda: Group.query.filter_by(type='dpt'),
        to_field_name='name',
        error_messages={'invalid_choice': 'Invalid destination'},
        widget=forms.TextInput(),
    )

    format = forms.ChoiceField(
        required=False,
        choices=(
            ('csv-txt', 'csv-txt'),
            ('csv', 'csv'),
            ('sql-txt', 'sql-txt'),
            ('sql', 'sql'),
        ))

    def get_message(self):
        return FormError(self).message
