import json

from django.http import HttpResponseForbidden, HttpResponse, \
    HttpResponseBadRequest
from django.shortcuts import render
from django.conf import settings
from django.utils.decorators import method_decorator, decorator_from_middleware
from django.views.generic import View

from django_yauth import middleware as yauth_middleware
from infra.cauth.server.common.models import User

from infra.cauth.server.master.api.views.base import RequestError
from infra.cauth.server.master.dashboard.models import Permission
from infra.cauth.server.master.utils.json_encoder import CustomEncoder


def yauth_decorator(f):
    if settings.YAUTH_ENABLED:
        mw_class = yauth_middleware.YandexAuthMiddleware
        decorator = decorator_from_middleware(mw_class)
        f = method_decorator(decorator)(f)
    return f


class DashboardBaseView(View):
    template_name = None
    content_type = None
    yalogin_required = True

    NAV = None

    def test_permissions(self, user_perms):
        raise NotImplementedError

    def get_permissions(self):
        login = self.request.yauser.login
        user = User.get_one(login=login)

        group_names = {g.name for g in user.groups}

        query = Permission.objects.filter(
            subject_name__in=group_names | {login})

        return {p.given_permission for p in query}

    @yauth_decorator
    def dispatch(self, request, *args, **kwargs):
        if settings.YAUTH_ENABLED and self.yalogin_required:
            mw = yauth_middleware.YandexAuthRequiredMiddleware()
            response = mw.process_request(request)
            if response is not None:
                return response

            if not self.test_permissions(self.get_permissions()):
                return HttpResponseForbidden("Forbidden.")

        try:
            result = super(DashboardBaseView, self).dispatch(request, *args,
                                                             **kwargs)
            response_cls = HttpResponse
        except RequestError as error:
            result = {
                'error': error.message,
            }
            response_cls = HttpResponseBadRequest

        if isinstance(result, dict):
            if self.content_type == 'json':
                content = json.dumps(result, cls=CustomEncoder,
                                     ensure_ascii=False)
                return response_cls(content, content_type='application/json')
            elif self.content_type == 'html':
                result['active_nav'] = self.NAV
                return render(request, self.template_name, result)

        return result
