import contextlib
import json
import os

from django_celery_beat.models import PeriodicTask

from django.conf import settings
from django.core.urlresolvers import reverse
from django.http import Http404, HttpResponseRedirect

from infra.cauth.server.master.celery_app import app
from infra.cauth.server.master.dashboard.views.base import DashboardBaseView
from infra.cauth.server.master.utils.mongo import get_mongo_database
from infra.cauth.server.master.utils.tasks import remove_lock_by_name

from infra.cauth.server.master.dashboard.forms import PeriodicTaskForm


class DashboardCeleryBase(DashboardBaseView):
    NAV = 'celery'

    def test_permissions(self, user_perms):
        return 'celery' in user_perms or 'all' in user_perms

    def dispatch(self, *args, **kwargs):
        app.autodiscover_tasks(settings.INSTALLED_APPS, force=True)
        return super(DashboardCeleryBase, self).dispatch(*args, **kwargs)


class DashboardCeleryLogView(DashboardCeleryBase):
    template_name = 'dashboard/celery-log.html'
    content_type = 'html'

    def get(self, request, id):
        db = get_mongo_database()

        log_record = db['task_logs'].find_one({'task_id': id})
        if not log_record:
            return {
                'task_id': id,
                'notfound': True,
            }

        filename = os.path.join(settings.TASKS_LOG_DIR, log_record['filename'])

        try:
            with contextlib.closing(open(filename)) as f:
                if request.GET.get('content_type') == 'json':
                    self.content_type = 'json'

                try:
                    offset = int(request.GET.get('offset', '0'))
                except ValueError:
                    offset = 0

                if offset:
                    f.seek(offset)

                content = f.read().decode('utf-8')

        except IOError:
            offset = 0
            content = ''

        if 'finished' in log_record:
            finished = log_record['finished'].isoformat()
        else:
            finished = None

        return {
            'task_id': id,
            'task_name': log_record['task_name'],
            'started': log_record['started'].isoformat(),
            'finished': finished,
            'content': content,
            'size': offset + len(content),
        }


class DashboardCeleryTaskView(DashboardCeleryBase):
    template_name = 'dashboard/celery-task.html'
    content_type = 'html'

    @staticmethod
    def format_schedule(obj):
        if obj.crontab:
            components = (
                obj.crontab.minute,
                obj.crontab.hour,
                obj.crontab.day_of_week,
                obj.crontab.day_of_month,
                obj.crontab.month_of_year,
            )
        elif obj.interval:
            components = (
                str(obj.interval.every),
                obj.interval.period,
            )
        else:
            raise ValueError("no crontab and not interval")

        return ' '.join(components)

    def get(self, request, id=None):
        return self.handle(request, id)

    def post(self, request, id=None):
        return self.handle(request, id)

    def handle(self, request, id=None):
        logs = []

        if id:
            try:
                pt = PeriodicTask.objects.get(id=id)
            except PeriodicTask.DoesNotExist:
                raise Http404
            schedule_string = self.format_schedule(pt)

            task = app.tasks.get(pt.task)
            if task and hasattr(task, 'info'):
                for log in task.info.bind(pt).get_recent_logs(30):
                    try:
                        stat = os.stat(os.path.join(settings.TASKS_LOG_DIR,
                                                    log['filename']))
                        filesize = stat.st_size

                        if filesize > 1024:
                            filesize = "{}k".format(filesize/1024)
                        else:
                            filesize = str(filesize)
                    except Exception:
                        filesize = "missing"

                    logs.append({
                        'id': log['task_id'],
                        'started': log['started'],
                        'size': filesize,
                        'url': reverse('dashboard-celery-log',
                                       kwargs={'id': log['task_id']}),
                    })
        else:
            schedule_string = ''
            pt = None

        if request.method == 'POST':
            form = PeriodicTaskForm(request.POST, instance=pt)
            schedule_string = request.POST.get('schedule', '')
            if form.is_valid():
                pt = form.save()
                return HttpResponseRedirect(reverse('dashboard-celery-task',
                                                    kwargs={'id': pt.id}))
        else:
            form = PeriodicTaskForm(instance=pt)

        return {
            'pt': form.instance,
            'form': form,
            'logs': logs,
            'schedule_value': schedule_string,
            'is_new': id is None,
        }


class DashboardCeleryUnlockView(DashboardCeleryBase):
    template_name = 'dashboard/celery-unlock.html'
    content_type = 'html'

    def post(self, request, id):
        try:
            pt = PeriodicTask.objects.get(id=id)
        except PeriodicTask.DoesNotExist:
            raise Http404

        if pt.task not in app.tasks:
            return {
                'error': 'Task "{}" is not registered in celery.'.format(
                    pt.task)
            }

        bound_info = app.tasks[pt.task].info.bind(pt)
        lock_info = bound_info.get_lock_info()

        if not lock_info:
            return {
                'success': 'Lock does not exist anymore.'
            }

        if lock_info['node'] != request.POST.get('node'):
            return {
                'error': 'Lock has changed. Try again.'
            }

        remove_lock_by_name(lock_info['node'])

        return {
            'success': 'Lock has been deleted.',
        }


class DashboardCeleryView(DashboardCeleryBase):
    template_name = 'dashboard/celery.html'
    content_type = 'html'

    def task_lock_context(self, bound_info):
        if not bound_info.info.use_lock:
            return {
                'use_lock': False,
                'active_lock': None,
            }

        return {
            'use_lock': True,
            'active_lock': bound_info.get_lock_info(),
        }

    def task_log_context(self, bound_info):
        latest_log = bound_info.get_latest_log()

        if latest_log:
            log_url = reverse('dashboard-celery-log',
                              kwargs={'id': latest_log['task_id']})
            return {
                'last_run': {
                    'timestamp': latest_log['started'],
                    'log_url': log_url,
                },
            }
        else:
            return {
                'last_run': None,
            }

    def get(self, request):
        tasks = []
        for pt in PeriodicTask.objects.all():
            task = app.tasks.get(pt.task)
            if not task or not hasattr(task, 'info'):
                continue

            bound_info = task.info.bind(pt)
            details_url = reverse('dashboard-celery-task', kwargs={'id': pt.id})

            task_context = {
                'id': pt.id,
                'name': pt.name,
                'task': pt.task,
                'enabled': pt.enabled,
                'schedule': str(pt.schedule),
                'details_url': details_url,
            }

            task_context.update(self.task_log_context(bound_info))
            task_context.update(self.task_lock_context(bound_info))

            tasks.append(task_context)

        return {'tasks': tasks}

    def post(self, request):
        pts = PeriodicTask.objects.filter(id__in=request.POST.getlist('task'))

        if 'enable' in request.POST:
            for pt in pts:
                pt.enabled = True
                pt.save()
        elif 'disable' in request.POST:
            for pt in pts:
                pt.enabled = False
                pt.save()
        elif 'remove' in request.POST:
            for pt in pts:
                pt.delete()
        elif 'run' in request.POST:
            for pt in pts:
                if pt.task in app.tasks:
                    task = app.tasks[pt.task]
                    task.apply_async(
                        args=json.loads(pt.args),
                        kwargs=json.loads(pt.kwargs),
                    )

        return HttpResponseRedirect('/dashboard/celery/')


index_view = DashboardCeleryView.as_view()
log_view = DashboardCeleryLogView.as_view()
task_view = DashboardCeleryTaskView.as_view()
unlock_view = DashboardCeleryUnlockView.as_view()
