import os
import json
import codecs
import urllib.request
import urllib.parse
import urllib.error

from django.http import Http404, HttpResponse, HttpResponseRedirect
from django.core.urlresolvers import reverse
from django.conf import settings

from infra.cauth.server.master.importers.registry import SuiteGroup
from infra.cauth.server.master.importers.tasks import run_import
from infra.cauth.server.master.utils.tasks import LockInfo
from infra.cauth.server.master.utils.tasks import remove_lock_by_name

from .base import DashboardBaseView
from ..forms import ImportsDownloadForm, ImportsDeleteForm, ImportUploadForm, \
    ImportUnlockForm


class DashboardImportsView(DashboardBaseView):
    template_name = 'dashboard/imports-index.html'
    content_type = 'html'

    NAV = 'imports'

    def test_permissions(self, user_perms):
        return 'imports' in user_perms or 'all' in user_perms

    def get(self, request):
        groups = []
        for group in list(SuiteGroup.registry.values()):
            original_filename = group.last_data_filename

            if original_filename:
                original_filename = os.path.basename(original_filename)
                original_url = '{url}?{query}'.format(
                    url=reverse('dashboard-imports-download'),
                    query=urllib.parse.urlencode({
                        'group': group.name,
                        'type': 'original',
                    })
                )
            else:
                original_url = None

            override_filename = group.override_filename
            if override_filename and os.path.isfile(override_filename):
                override_filename = os.path.basename(override_filename)
                override_url = '{url}?{query}'.format(
                    url=reverse('dashboard-imports-download'),
                    query=urllib.parse.urlencode({
                        'group': group.name,
                        'type': 'override',
                    })
                )
            else:
                override_filename = override_url = None

            active_locks = []
            for suite in list(group.suites.values()):
                lock_info = LockInfo(suite.get_lock_name()).get()
                if lock_info:
                    active_locks.append({
                        'suite': suite.TARGET,
                        'node': lock_info['node'],
                        'data': lock_info['data'],
                    })

            groups.append({
                'name': group.name,
                'original_filename': original_filename,
                'original_url': original_url,
                'override_filename': override_filename,
                'override_url': override_url,
                'active_locks': active_locks,
            })

        groups.sort(key=lambda item: item['name'])
        return {
            'groups': groups,
            'allow_uploading': bool(settings.IMPORTS_OVERRIDE_DIR),
            'yt_locks_ui_url': settings.YT_LOCKS_UI_URL,
        }


class ImportsDownloadView(DashboardBaseView):
    def test_permissions(self, user_perms):
        return 'imports' in user_perms or 'all' in user_perms

    def get(self, request):
        form = ImportsDownloadForm(request.GET)
        if not form.is_valid():
            raise Http404

        group = form.cleaned_data['group']

        if form.cleaned_data['type'] == 'original':
            filename = group.last_data_filename
        elif form.cleaned_data['type'] == 'override':
            filename = group.override_filename
        else:
            raise Http404

        if not filename:
            return HttpResponse("Not available.")

        response = HttpResponse(open(filename).read(),
                                content_type='application/json')
        response['Content-Disposition'] = 'attachment; filename={}'\
            .format(os.path.basename(filename))
        return response


class ImportsDeleteView(DashboardBaseView):
    def test_permissions(self, user_perms):
        return (bool(settings.IMPORTS_OVERRIDE_DIR)
                and ('imports' in user_perms or 'all' in user_perms))

    def post(self, request):
        form = ImportsDeleteForm(request.POST)
        if form.is_valid():
            group = form.cleaned_data['group']
            filename = group.override_filename

            if os.path.exists(filename):
                os.unlink(filename)

        return HttpResponseRedirect('/dashboard/imports/')


class ImportsRunView(DashboardBaseView):
    def test_permissions(self, user_perms):
        return 'imports' in user_perms or 'all' in user_perms

    def post(self, request):
        form = ImportsDeleteForm(request.POST)
        if form.is_valid():
            group = form.cleaned_data['group']
            result = run_import.delay(group.name, insane_mode=True)
            return HttpResponseRedirect('/dashboard/celery/{}/'.format(
                result.id))

        return HttpResponseRedirect('/dashboard/imports/')


class ImportsUnlockView(DashboardBaseView):
    template_name = 'dashboard/imports-unlock.html'
    content_type = 'html'

    NAV = 'imports'

    def test_permissions(self, user_perms):
        return 'imports' in user_perms or 'all' in user_perms

    def post(self, request):
        form = ImportUnlockForm(request.POST)
        if not form.is_valid():
            raise Http404

        suite = form.cleaned_data['suite']
        lock_info = LockInfo(suite.get_lock_name()).get()

        if not lock_info:
            return {
                'success': 'Lock does not exist anymore.'
            }

        if lock_info['node'] != request.POST.get('node'):
            return {
                'error': 'Lock has changed. Try again.'
            }

        remove_lock_by_name(lock_info['node'])

        return {
            'success': 'Lock has been deleted.',
        }


class ImportsUploadView(DashboardBaseView):
    content_type = 'html'
    template_name = 'dashboard/imports-upload.html'

    NAV = 'imports'

    def test_permissions(self, user_perms):
        return (bool(settings.IMPORTS_OVERRIDE_DIR)
                and ('imports' in user_perms or 'all' in user_perms))

    def post(self, request):
        form = ImportUploadForm(request.POST, request.FILES)
        if not form.is_valid():
            if 'group' in form.errors:
                raise Http404

            return {
                'group': form.cleaned_data['group'],
                'errors': form.errors['override'][0],
            }

        group = form.cleaned_data['group']
        filename = group.override_filename

        override_data = form.cleaned_data['override']
        json.dump(override_data, codecs.open(filename, 'w', encoding='utf-8'),
                  ensure_ascii=False, indent=2)

        result = run_import.delay(group.name, insane_mode=True)
        return HttpResponseRedirect('/dashboard/celery/{}/'.format(result.id))


index_view = DashboardImportsView.as_view()
download_view = ImportsDownloadView.as_view()
delete_view = ImportsDeleteView.as_view()
upload_view = ImportsUploadView.as_view()
run_view = ImportsRunView.as_view()
unlock_view = ImportsUnlockView.as_view()
