from django.core.files.base import ContentFile
from django.db import models

from infra.cauth.server.master.constants import FILE_TYPE
from infra.cauth.server.master.files.storage import CAuthS3Boto3Storage
from infra.cauth.server.master.files.managers import S3FileManager


class S3File(models.Model):
    type = models.CharField(max_length=64, choices=FILE_TYPE.choices, default=FILE_TYPE.OTHER, db_index=True)
    name = models.CharField(max_length=100, db_index=True, null=True, blank=True)
    file = models.FileField(storage=CAuthS3Boto3Storage())
    created_at = models.DateTimeField(auto_now_add=True, db_index=True)

    objects = S3FileManager()

    @classmethod
    def get_obj_data(cls, filename):
        return cls.objects.get(name=filename).file.read()

    @classmethod
    def create_obj(cls, name, file_type, data):
        if isinstance(data, str):
            data = data.encode()
        obj = cls(type=file_type)
        obj.file.save(name, content=ContentFile(data), save=False)
        obj.name = obj.file.name
        try:
            obj.save()
        except Exception:
            obj.file.storage.delete(obj.name)
            raise

        return obj

    def __repr__(self):
        creation_date = (
            self.created_at.strftime('%Y-%m-%d %H:%M:%S')
            if self.created_at
            else 'not saved'
        )
        return '<{}: [{}] {} ({})>'.format(
            type(self).__name__,
            self.type,
            self.file.name or 'None',
            creation_date,
        )
