import collections
import datetime
import json
import uuid

from django.conf import settings

from infra.cauth.server.common.models import ImportStatus
from infra.cauth.server.common.alchemy import Session

from infra.cauth.server.master.constants import FILE_TYPE
from infra.cauth.server.master.files.models import S3File
from infra.cauth.server.master.utils.tasks import lock_manager
from infra.cauth.server.master.utils.subtasks import SubtaskPool, FakeSubtaskPool
from infra.cauth.server.master.utils.config import get_class_path

from .tasks import run_import_subtask

# TODO(lavrukov): унифицировать данные в to_*
AnalyzeResult = collections.namedtuple('AnalyzeResult', (
    'to_add',
    'to_remove',
    'to_update',
))


class GeneralImporter(object):
    """
    Importer that loads whole datasets into memory, analyzes them, makes
    incremental changes.

    Will be unsuitable for extremely large data sets.
    """

    TARGET = None
    suite = None  # updated by suite metaclass
    subtask_classes = {}
    sanity_limits = {'any': 2000}

    __item_attrs__ = None  # Attributes used for checking equality/updating

    def __init__(self, suite, instream, outstream, logger, do_sanity_checks=True,
                 whats_new=False):
        self.instream = instream
        self.outstream = outstream
        self.do_sanity_checks = do_sanity_checks
        self.whats_new = whats_new

        self.suite = suite
        self.logger = logger

        self._subtask_pool = None

    def load_existing_data(self):
        """
        Loads currently existing data in format {id: item, <...>},
        where id is some kind of id or hash that unambigiously identifies an
        item both for existing and new data, and item is full information on
        entity data (preferably a namedtuple)
        """
        raise NotImplementedError

    def load_new_data(self, input):
        """
        Same semantics load_existing_data, but for new data, obviously
        """
        raise NotImplementedError

    @property
    def subtask_pool(self):
        if self._subtask_pool is None:
            pool_cls = SubtaskPool if settings.CAUTH_ASYNCHRONOUS_SYNC else FakeSubtaskPool
            self._subtask_pool = pool_cls(suite_run_id=self.suite.run_id)

        return self._subtask_pool

    def add_subtask(self, name, data):
        if name not in self.subtask_classes:
            raise ValueError("Unknown subtask name: {}".format(name))

        subtask_cls = self.subtask_classes[name]
        filename = '_'.join([self.suite.run_id, self.suite.group.name, name, 'subtask'])

        s3_file = S3File.create_obj(filename, file_type=FILE_TYPE.SUBTASK, data=json.dumps(data))
        self.subtask_pool.apply(
            task=run_import_subtask,
            time_limit=subtask_cls.time_limit,
            args=[],
            kwargs={
                'cls_path': get_class_path(subtask_cls),
                'filename': s3_file.name,
            },
        )

    @classmethod
    def diff(cls, first, second):
        """
        Return diference between items
        as three-tuples: ((attribute, old, new), <...>)
        or None if they are equal
        """
        res = []
        for att in cls.__item_attrs__:
            val_first = getattr(first, att)
            val_second = getattr(second, att)
            if val_first != val_second:
                res.append((att, val_first, val_second))

        return tuple(res) or None

    def validate_new_data(self):
        self.instream.seek(0)
        data = json.load(self.instream)
        self.suite.group.validator().validate(data)
        return data

    def load_data(self):
        new_data = self.validate_new_data()

        Session.remove()

        self.logger.info("Loading existing data")
        self.ex_data = self.load_existing_data()
        self.logger.info("Loading new data")
        self.new_data = self.load_new_data(new_data)

        self.logger.info("Existing: %s, new: %s",
                         len(self.ex_data), len(self.new_data))

    def post_import(self):
        pass

    def run_import(self):
        self.logger.info("Starting importer %s (suite_run_id=%s)",
                         self.__class__.__name__, self.suite.run_id)

        self.load_data()

        res = self.res = self.analyze(self.ex_data, self.new_data)
        self.logger.info("To add: %s, to delete: %s, to update: %s",
                         len(res.to_add), len(res.to_remove),
                         len(res.to_update))

        if self.whats_new:
            self.print_whats_new()
            return

        if self.do_sanity_checks:
            self.sanity_checks(res)

        if res.to_remove:
            self.remove(res.to_remove)
            self.logger.info("Deleted %s", len(res.to_remove))
        if res.to_add:
            self.add(res.to_add)
            self.logger.info("Added %s", len(res.to_add))
        if res.to_update:
            self.update(res.to_update)
            self.logger.info("Updated %s", len(res.to_update))

        self.post_import()

    def sanity_checks(self, res):
        """
        A hook used to check analyze results before doing the operations.
        For example, if you have too many deletes, you may want to crash import
        process and check if it is indeed ok.
        """
        for op in ('add', 'remove', 'update'):
            value = getattr(res, 'to_{}'.format(op))

            limit = self.sanity_limits.get(op) or self.sanity_limits.get('any')

            if limit is not None:
                count = len(value)
                assert count <= limit, "%s is off the limits (%s > %s)" % (op, count, limit)

    def print_whats_new(self):
        def print_res_part(part):
            for pair in part.items():
                print("    %s: %s" % pair)
        if self.res.to_add:
            print("Entries to add:")
            print_res_part(self.res.to_add)
        if self.res.to_update:
            print("Entries to update:")
            print_res_part(self.res.to_update)

        if self.res.to_remove:
            print("Entries to remove:")
            print_res_part(self.res.to_remove)

    def add(self, keys):
        # TODO(lavrukov): нужно переименовать keys в to_update, т.к нам приходит
        # именно дикт, а не список ключей. То же нужно повторить для remove и
        # update
        raise NotImplementedError

    def remove(self, keys):
        raise NotImplementedError

    def update_one(self, key):
        """ Update one entry """
        raise NotImplementedError

    def update(self, keys):
        """For large amounts of updates, selecting and updating one-by-one may
        be ineffective. If this is ever a problem, it is easy to refactor.
        """
        for key, diff in self.res.to_update.items():
            self.update_one(key)

    @classmethod
    def analyze(cls, ex_data, new_data):
        ex_keys = set(ex_data.keys())
        new_keys = set(new_data.keys())

        keys_to_add = new_keys - ex_keys
        keys_to_remove = ex_keys - new_keys

        to_add = {key: new_data[key] for key in keys_to_add}
        to_remove = {key: ex_data[key] for key in keys_to_remove}

        other_keys = ex_keys & new_keys
        to_update = {}
        for key in other_keys:
            diff = cls.diff(ex_data[key], new_data[key])
            if diff:
                to_update[key] = diff

        return AnalyzeResult(
            to_add=to_add,
            to_update=to_update,
            to_remove=to_remove
        )


class ImportSuiteMeta(type):
    def __new__(mcs, *args):
        new_cls = type.__new__(mcs, *args)

        for importer_cls in new_cls.IMPORTER_CLASSES:
            importer_cls.suite = new_cls

        return new_cls


class ImportSuite(object, metaclass=ImportSuiteMeta):
    IMPORTER_CLASSES = ()
    TARGET = None

    group = None  # updated by suite registry

    def __init__(self, logger, insane_mode=False):
        self.run_id = str(uuid.uuid4())
        self.logger = logger
        self.out_stream = None
        self.insane_mode = insane_mode
        self.lock = None

    @classmethod
    def get_lock_name(cls):
        return ".".join(("importers", cls.group.name, cls.TARGET))

    @property
    def override_filename(self):
        return self.group.override_filename

    def acquire_lock(self):
        if settings.DISABLE_TASK_LOCKING:
            return True
        else:
            self.lock = lock_manager.lock(self.get_lock_name(), block=False)
            return self.lock.acquire()

    def release_lock(self):
        if not settings.DISABLE_TASK_LOCKING:
            self.lock.release()

    def post_run(self, instream):
        q = ImportStatus.query.filter_by(suite=self.group.name,
                                         target=self.TARGET)

        status = q.first()

        if not status:
            status = ImportStatus(
                suite=self.group.name,
                target=self.TARGET,
            )
            Session.add(status)

        status.last_import = datetime.datetime.now()
        Session.commit()

    def run(self, in_filename):
        self.logger.info('Starting import suite %s (run_id=%s)',
                         type(self).__name__, self.run_id)

        in_file = S3File.objects.get(name=in_filename)
        for importer_cls in self.IMPORTER_CLASSES:
            name = importer_cls.__name__
            self.logger.info('Starting importer: %s (using %s)', name, in_filename)

            try:
                importer = importer_cls(
                    suite=self,
                    instream=in_file.file,
                    outstream=self.out_stream,
                    do_sanity_checks=not self.insane_mode,
                    logger=self.logger,
                )
            except Exception:
                self.logger.exception("Failed to initialize importer %s", name)
                raise

            try:
                importer.run_import()
            except Exception:
                self.logger.exception("Failed to run importer %s", name)
                raise

            self.logger.info("Importer complete: %s", name)

        try:
            self.post_run(in_file.file)
        except Exception:
            self.logger.exception("Failure in post_run section of %s suite",
                                  self.__class__.__name__)
            raise
