import collections
import json

from marshmallow import ValidationError

from infra.cauth.server.master.importers import registry
from infra.cauth.server.master.files.models import S3File


class CustomEncoder(json.JSONEncoder):
    def default(self, o):
        if isinstance(o, set):
            return list(sorted(o))

        return super(CustomEncoder, self).default(o)

    def encode(self, o):
        if isinstance(o, dict) and not isinstance(o, collections.OrderedDict):
            o = collections.OrderedDict((k, o[k]) for k in sorted(o.keys()))
        return super(CustomEncoder, self).encode(o)


def fetch_group(group, filename, logger):
    func = registry.SuiteGroup.registry[group].fetcher

    try:
        data = func(logger)
    except (ValidationError, ValueError) as exc:
        logger.error("Validation failed with error: %s", str(exc))
        raise

    json_data = json.dumps(data, cls=CustomEncoder, indent=2, ensure_ascii=False)
    file_type = group + '_history'
    s3_file = S3File.create_obj(filename, file_type=file_type, data=json_data)

    logger.info("Download and validation OK, saved to file: %s", s3_file.name)
