import datetime
import logging

from django.core.management.base import BaseCommand

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.constants import IDM_STATUS
from infra.cauth.server.common.models import Server, ServerGroup
from infra.cauth.server.master.api.idm.update import ensure_remote_nodes_correct
from infra.cauth.server.master.constants import BATCH_OPERATION


log = logging.getLogger(__name__)


class Command(BaseCommand):
    def add_arguments(self, parser):
        parser.add_argument('dst', type=str)

    def handle(self, dst, **options):
        server_query = Server.query.filter_by(fqdn=dst)
        group_query = ServerGroup.query.filter_by(name=dst)
        dst_obj = server_query.first() or group_query.first() or dst

        if isinstance(dst_obj, Server):
            obj_type = 'server'
        elif isinstance(dst_obj, ServerGroup):
            obj_type = 'group'
        else:
            obj_type = 'nothing'

        log.info(
            'Found {} with dst {}, comparing local rolenodes with IDM rolenodes'.format(
                obj_type, dst
            ))

        considered_actual = False
        if obj_type != 'nothing' and dst_obj.idm_status == IDM_STATUS.ACTUAL:
            considered_actual = True
            log.warning('WARNING: trying to push dst {} which is considered actual'.format(dst))

        was_pushed = ensure_remote_nodes_correct(dst_obj, BATCH_OPERATION.PUSH_DST_COMMAND)

        if obj_type != 'nothing' and dst_obj.first_pending_push_started_at:
            dst_obj.last_push_ended_at = datetime.datetime.now()

        if was_pushed and considered_actual:
            error = 'dst {} was considered actual, but actually was not'.format(dst)
            log.warning(error)

        if not was_pushed:
            log.info('Did not push dst {} to IDM: empty diff found'.format(dst))
            if obj_type != 'nothing' and dst_obj.idm_status != IDM_STATUS.ACTUAL:
                dst_obj.idm_status = IDM_STATUS.ACTUAL
                dst_obj.became_actual_at = datetime.datetime.now()
                Session.commit()
                log.info('Dst {} was marked as actual'.format(dst))
