import collections
import datetime

from infra.cauth.server.common.models import User
from django.conf import settings

from . import sources


class ServersGroupsAggregator(object):
    server_sources = (
        sources.GolemServers,
    )

    group_sources = (
        sources.ConductorGroups,
        sources.GencfgGroups,
        sources.YpGroups,
        sources.WalleProjects,
        sources.BotAbcGroups,
        sources.HdAbcGroups,
    )

    # Сейчас порядок определяет тип сервера (выше <-> приоритетней)
    # YP точно должен быть первым, чтобы dev-server не перезаписался на server,
    # который часто приезжает из conductor'а
    source_priority = (
        sources.YpGroups,
        sources.WalleProjects,
        sources.GencfgGroups,
        sources.ConductorGroups,
        sources.BotAbcGroups,
        # ===== сейчас хосты не из групп не имеют приоритета (но группы важнее) =====
        sources.GolemServers,
        sources.HdAbcGroups,
    )
    source_name_priority = [
        (getattr(source, 'NAME', None) or getattr(source, 'PREFIX', None))
        for source in source_priority
    ]

    @classmethod
    def source_sort_key(cls, obj):
        index = cls.source_name_priority.index(obj['source'])
        return index, obj['name']

    def __init__(self, logger):
        self._logger = logger
        self._server_map = {}
        self._group_list = []
        self._successful_sources = set()
        self._active_logins = {u.login for u
                               in User.query.filter_by(is_fired=False)}
        self.active_server_sources = [
            source for source in self.server_sources if source.NAME in settings.SERVER_SOURCES
        ]
        self.active_group_sources = [
            source for source in self.group_sources if source.PREFIX in settings.GROUP_SOURCES
        ]

    def get_server(self, name):
        return self._server_map[name]

    def only_active_logins(self, logins):
        return {login for login in logins if login in self._active_logins}

    def _load_servers(self):
        for src_cls in self.active_server_sources:
            src = src_cls(logger=self._logger)

            if src_cls.NAME in settings.SUSPENDED_SOURCES:
                source_data = src.fetch_from_database()
            else:
                try:
                    source_data = src.fetch()
                    self._successful_sources.add(src_cls.NAME)
                except Exception:
                    self._logger.exception("Failed to fetch "
                                           "server source {}".format(src_cls.NAME))
                    source_data = src.fetch_from_database()

            for srv in source_data:
                if srv.name in self._server_map:
                    target = self._server_map[srv.name]
                else:
                    target = srv.as_ordered_dict()
                    self._server_map[srv.name] = target
                    # responsibles are determined per source
                    target['responsibles'] = collections.defaultdict(set)

                responsibles = self.only_active_logins(srv.responsibles)
                target['responsibles'][srv.source].update(responsibles)

    def _load_groups(self):
        for src_cls in self.active_group_sources:
            src = src_cls(logger=self._logger)

            if src_cls.PREFIX in settings.SUSPENDED_SOURCES:
                source_data = src.fetch_from_database()
            else:
                try:
                    source_data = src.fetch()
                    self._successful_sources.add(src_cls.PREFIX)
                except Exception:
                    self._logger.exception("Failed to fetch "
                                           "server group source {}".format(src_cls.PREFIX))
                    source_data = src.fetch_from_database()

            for grp in source_data:
                serialized = grp.as_ordered_dict()
                serialized['responsibles'] = self.only_active_logins(grp.responsibles)
                self._group_list.append(serialized)

    def fetch(self):
        timestamp = datetime.datetime.now()

        self._load_servers()
        self._load_groups()

        result = collections.OrderedDict((
            ('servers', list(self._server_map.values())),
            ('groups', self._group_list),
            ('updated_sources', list(self._successful_sources)),
            ('timestamp', timestamp.isoformat()),
        ))

        result['servers'].sort(key=lambda s: s['name'])
        result['groups'].sort(key=lambda g: self.source_sort_key(g))
        result['updated_sources'].sort()

        return result


def fetch_servers(logger):
    return ServersGroupsAggregator(logger).fetch()
