import abc
import json
import logging
import urllib.error
import urllib.request
import urllib.parse
from collections import OrderedDict, defaultdict

import attr
import lxml.etree
from django.conf import settings
from django.utils.text import force_text
from ids.registry import registry
from ids.services.abc.connector import ABCConnector
from ids.services.abc.repositories import ABCRepository
from typing import Set, List, AnyStr, Dict, Iterable, Optional
from sqlalchemy.orm import joinedload

from infra.cauth.server.common.constants import SERVER_TYPE
from infra.cauth.server.common.models import Source, Server, ServerGroup, User

from infra.cauth.server.master.utils.http_client import HttpClient
from infra.cauth.server.master.utils.fqdn import should_be_added

log = logging.getLogger(__name__)

instance_of = attr.validators.instance_of

ABCConnector.url_patterns['resource_consumers'] = '/api/v4/resources/consumers/'


@registry.add_simple
class ResourceConsumersRepository(ABCRepository):

    SERVICE = 'abc'
    RESOURCES = 'resource_consumers'


def all_instance_of(type_):
    return attr.validators.deep_iterable(instance_of(type_))


@attr.s
class ServerResult(object):
    name = attr.ib(type=AnyStr, validator=instance_of(str))
    source = attr.ib(type=AnyStr, validator=instance_of(str))
    type = attr.ib(type=AnyStr, default=SERVER_TYPE.SERVER, validator=instance_of(str))
    is_baremetal = attr.ib(type=bool, default=False, validator=instance_of(bool))
    responsibles = attr.ib(type=Set, factory=set, validator=all_instance_of(str))

    def as_ordered_dict(self):
        return OrderedDict((
            ('name', self.name),
            ('type', self.type),
            ('responsibles', self.responsibles),
            ('is_baremetal', self.is_baremetal)
        ))


class ServerSourceBase(object):
    NAME = None
    URL = None

    def __init__(self, logger):
        self._logger = logger

    def fetch(self):
        # typing: () -> List[ServerResult]
        data = force_text(HttpClient.fetch(self.URL, logger=self._logger, timeout=60))

        results = []
        for line in data.splitlines():
            line = line.strip()
            if not line or line[:1] == '#':
                continue

            try:
                name, responsibles = line.lower().split('\t', 1)
            except ValueError:
                continue

            item = ServerResult(
                name=name.strip(),
                source=self.NAME,
                responsibles={r.strip() for r in responsibles.strip().split(',') if r.strip()},
            )
            if should_be_added(item.name):
                results.append(item)

        return results

    def fetch_from_database(self):
        # typing: () -> List[ServerResult]
        source = Source.get_one(name=self.NAME)
        if not (source and source.last_update):
            raise RuntimeError(
                "Source {} has never been updated.".format(self.NAME))

        uid2login = {u.uid: u.login for u in User.query}
        query = Server.query\
            .join('sources')\
            .filter(Source.id == source.id)

        results = []
        for s in query:
            item = ServerResult(
                name=s.fqdn,
                type=s.type,
                responsibles={uid2login[r.uid] for r in s.responsibles if r.source_id == source.id},
                source=self.NAME,
            )

            if should_be_added(item.name):
                results.append(item)

        return results


@attr.s
class GroupResult(object):
    @attr.s
    class Host(object):
        hostname = attr.ib(type=AnyStr, validator=instance_of(str))
        type = attr.ib(type=AnyStr, default=SERVER_TYPE.SERVER, validator=instance_of(str))
        is_baremetal = attr.ib(type=bool, default=False, validator=instance_of(bool))

    @attr.s
    class Contacts(object):
        request_email = attr.ib(
            type=Optional[AnyStr],
            default=None,
            validator=instance_of((str, type(None))),
        )
        request_queue = attr.ib(
            type=Optional[AnyStr],
            default=None,
            validator=instance_of((str, type(None))),
        )
        notify_email = attr.ib(
            type=Optional[AnyStr],
            default=None,
            validator=instance_of((str, type(None))),
        )

    @attr.s
    class Settings(object):
        flow = attr.ib(type=AnyStr, default=None, validator=instance_of((str, type(None))))
        trusted_sources = attr.ib(type=Set[AnyStr], factory=set, validator=all_instance_of(str))

    @attr.s
    class KeysInfo(object):
        key_sources = attr.ib(type=Set[AnyStr], factory=set, validator=all_instance_of(str))
        secure_ca_list_url = attr.ib(type=AnyStr, default=None, validator=instance_of((str, type(None))))
        insecure_ca_list_url = attr.ib(type=AnyStr, default=None, validator=instance_of((str, type(None))))
        krl_url = attr.ib(type=AnyStr, default=None, validator=instance_of((str, type(None))))
        sudo_ca_list_url = attr.ib(type=AnyStr, default=None, validator=instance_of((str, type(None))))

    source = attr.ib(type=AnyStr, validator=instance_of(str))
    name = attr.ib(type=Optional[AnyStr], default=None, validator=instance_of((str, type(None))))
    contacts = attr.ib(type=Optional[Contacts], factory=Contacts)
    hosts = attr.ib(type=List[Host], factory=list)
    responsibles = attr.ib(type=Set[AnyStr], factory=set, validator=all_instance_of(str))
    settings = attr.ib(type=Optional[Settings], factory=Settings)
    keys_info = attr.ib(type=Optional[KeysInfo], factory=KeysInfo)

    def as_ordered_dict(self):
        assert self.name
        return OrderedDict((
            ('name', self.name),
            ('source', self.source),
            ('responsibles', self.responsibles),
            ('hosts', list(map(attr.asdict, self.hosts))),
            ('contacts', OrderedDict((
                ('request_email', self.contacts.request_email),
                ('request_queue', self.contacts.request_queue),
                ('notify_email', self.contacts.notify_email),
            ))),
            ('settings', OrderedDict((
                ('flow', self.settings.flow),
                ('trusted_sources', self.settings.trusted_sources),
            ))),
            ('keys_info', OrderedDict((
                ('key_sources', self.keys_info.key_sources),
                ('secure_ca_list_url', self.keys_info.secure_ca_list_url),
                ('insecure_ca_list_url', self.keys_info.insecure_ca_list_url),
                ('krl_url', self.keys_info.krl_url),
                ('sudo_ca_list_url', self.keys_info.sudo_ca_list_url),
            ))),
        ))


class GroupSourceBase(object):
    PREFIX = None

    VALIDATORS = (
        lambda g: g.name and len(g.name) > 0,
    )

    def __init__(self, logger):
        self._logger = logger
        self._groups_map = {}  # typing: Dict[AnyStr, GroupResult]

    @classmethod
    def _qualify_name(cls, name):
        return '.'.join((cls.PREFIX, name))

    @classmethod
    def _is_valid(cls, group):
        # typing: (GroupResult) - > bool
        return all([f(group) for f in cls.VALIDATORS])

    def load_hosts(self):
        pass

    def load_responsibles(self):
        pass

    def load_settings(self):
        pass

    def fetch(self):
        # typing: () -> List[GroupResult]
        self.load_hosts()
        self.load_responsibles()
        self.load_settings()

        return list(filter(self._is_valid, list(self._groups_map.values())))

    def fetch_from_database(self):
        # typing: () -> List[GroupResult]
        source = Source.get_one(name=self.PREFIX)
        if not (source and source.last_update):
            raise RuntimeError(
                "Source {} has never been updated.".format(self.PREFIX))

        query = ServerGroup.query.options(
            joinedload(ServerGroup.servers).load_only(Server.fqdn, Server.type, Server.is_baremetal),
            joinedload(ServerGroup.responsible_users).load_only(User.login),
            joinedload(ServerGroup.trusted_sources).load_only(Source.name),
        ).filter(ServerGroup.source_id == source.id)

        results = []
        for g in query:
            item = GroupResult(
                name=g.name,
                responsibles={u.login.lower() for u in g.responsible_users},
                source=self.PREFIX,
                hosts=[
                    GroupResult.Host(hostname=s.fqdn.lower(), type=s.type, is_baremetal=s.is_baremetal)
                    for s in g.servers
                ],
                contacts=GroupResult.Contacts(
                    request_email=g.email.lower() if g.email else None,
                    request_queue=g.request_queue.upper() if g.request_queue else None,
                    notify_email=g.notify_email.lower() if g.notify_email else None,
                ),
                settings=GroupResult.Settings(
                    flow=g.flow,
                    trusted_sources={s.name for s in g.trusted_sources},
                ),
                keys_info=GroupResult.KeysInfo(
                    key_sources={source for source in g.key_sources.split(',')} if g.key_sources else set(),
                    secure_ca_list_url=g.secure_ca_list_url,
                    insecure_ca_list_url=g.insecure_ca_list_url,
                    krl_url=g.krl_url,
                    sudo_ca_list_url=g.sudo_ca_list_url,
                )
            )

            if self._is_valid(item):
                results.append(item)

        return results


class GolemServers(ServerSourceBase):
    NAME = 'golem'
    URL = 'http://ro.admin.yandex-team.ru/api/get_all_resp.sbml?get=do'


@attr.s
class AbcService(object):
    id_ = attr.ib(type=int)
    slug = attr.ib(type=AnyStr)
    hardware_resps = attr.ib(type=Set[AnyStr], factory=set)
    head = attr.ib(type=AnyStr, default=None)


class AbcGroups(GroupSourceBase):

    def __init__(self, logger):
        super(AbcGroups, self).__init__(logger)
        self._services = {}  # type: Dict[int, AbcService]
        self._abc_repo = registry.get_repository(
            service='abc',
            resource_type='service_members',
            user_agent=settings.USER_AGENT,
            oauth_token=settings.OAUTH_TOKEN,
        )

    def _iter_abc(self, lookup):
        # type: (Dict) -> Iterable[Dict]
        for item in self._abc_repo.getiter(lookup=lookup):
            yield item

    def _get_or_create_group(self, abc_id):
        # type: (str) -> GroupResult
        abc_id = int(abc_id)
        group = self._groups_map.get(abc_id)
        if group is None:
            self._groups_map[abc_id] = group = GroupResult(self.PREFIX)
        return group

    def _get_service_roles(self, service_ids=None):
        # type: (List) -> Iterable[Dict]
        lookup = {
            'role__in': ','.join(map(str, [self.HARDWARE_MANAGER_ID, self.PRODUCT_HEAD_ID])),
            'fields': 'role.id,service.id,service.slug,person.login',
        }
        if service_ids is None:
            service_ids = list(self._groups_map)
        for pos in range(0, len(service_ids), self.SERVICES_PER_REQUEST):
            cur_service_ids = ','.join(map(str, service_ids[pos:pos+self.SERVICES_PER_REQUEST]))
            cur_lookup = dict(service__in=cur_service_ids, **lookup)
            for member in self._iter_abc(cur_lookup):
                yield member

    def _load_services(self, services_ids=None):
        for member in self._get_service_roles(services_ids):
            id_ = member['service']['id']
            service_obj = self._services.get(id_)
            if service_obj is None:
                self._services[id_] = service_obj = AbcService(
                    id_=id_,
                    slug=member['service']['slug']
                )
            login = member['person']['login']
            role_id = member['role']['id']
            if role_id == self.HARDWARE_MANAGER_ID:
                service_obj.hardware_resps.add(login)
            elif role_id == self.PRODUCT_HEAD_ID:
                service_obj.head = login

    def _is_both_not_none(self, fqdn, abc_id):
        # type : (int, int) -> (bool)

        if not (fqdn and abc_id):
            if fqdn:
                self._logger.warning('FQDN {} has no abc id'.format(fqdn))
            return False
        return True

    def load_responsibles(self):
        for id_, service_obj in list(self._services.items()):
            resps = service_obj.hardware_resps
            if service_obj.head:
                resps.add(service_obj.head)
            self._groups_map[id_].responsibles = resps


class BotAbcGroups(AbcGroups):
    PREFIX = 'bot'

    HOSTS_URL = urllib.parse.urljoin(settings.BOT_URL, '/api/view.php?name=view_oops_hardware')

    HARDWARE_MANAGER_ID = 742
    PRODUCT_HEAD_ID = 1

    SERVICES_PER_REQUEST = 10

    def _get_servers_from_bot(self):
        # type: () -> AnyStr
        return HttpClient.fetch(self.HOSTS_URL, logger=self._logger, timeout=900)

    def load_hosts(self):
        for line in self._get_servers_from_bot().decode().splitlines():
            data = line.split('\t')
            fqdn, abc_id = data[1], data[-2]

            if not self._is_both_not_none(fqdn, abc_id):
                continue

            group = self._get_or_create_group(abc_id)
            host = GroupResult.Host(fqdn.strip(), is_baremetal=True)
            group.hosts.append(host)
        self._load_services()
        for id_, service_obj in list(self._services.items()):
            self._groups_map[id_].name = self._qualify_name(service_obj.slug)


class HdAbcGroups(AbcGroups):
    PREFIX = 'hd'

    HOST_URL = urllib.parse.urljoin(settings.HD_URL, '/api/v1/device/zombies')

    HARDWARE_MANAGER_ID = settings.HD_ABC_MANAGER_ID
    PRODUCT_HEAD_ID = 1
    ZOMBIE_TYPE = settings.ZOMBIE_TYPE

    SERVICES_PER_REQUEST = 10
    IDS_PER_REQUEST = 30

    def __init__(self, logger):
        super(HdAbcGroups, self).__init__(logger)

        self._abc_repo_consumers = registry.get_repository(
            service='abc',
            resource_type='resource_consumers',
            user_agent=settings.USER_AGENT,
            oauth_token=settings.OAUTH_TOKEN,
        )

    def _iter_consumers(self, lookup):
        # type: (Dict) -> Iterable[Dict]
        for item in self._abc_repo_consumers.getiter(lookup=lookup):
            yield item

    def _get_abc_ids_by_logins(self, logins):
        # type: (List[AnyStr]) -> Dict[AnyStr, int]
        lookup = {
            'fields': 'resource.external_id,service.id',
            'type__in': self.ZOMBIE_TYPE
        }

        fqdn_abc_id = defaultdict(list)

        for pos in range(0, len(logins), self.IDS_PER_REQUEST):
            cur_service_ids = ','.join(map(str, logins[pos:pos+self.IDS_PER_REQUEST]))
            cur_lookup = dict(resource__external_id__in=cur_service_ids, **lookup)
            for entity in self._iter_consumers(cur_lookup):
                fqdn_abc_id[entity['resource']['external_id']].append(entity['service']['id'])

        return fqdn_abc_id

    def _fetch_hd_data(self):
        # type: () -> AnyStr

        return HttpClient.fetch(self.HOST_URL, logger=self._logger, timeout=60, oauth=True)

    def _get_data_from_hd(self):
        # type: () -> Dict

        json_data = self._fetch_hd_data()

        try:
            data = json.loads(json_data)
        except Exception:
            self._logger.exception("Failed to decode json with hosts")
            raise

        return {instance['fqdn'].strip(): instance['login'].strip() for instance in data}

    def load_hosts(self):
        data = self._get_data_from_hd()
        logins = list(set(data.values()))
        fqdn_abc_id = self._get_abc_ids_by_logins(logins)

        for fqdn in data:
            login = data[fqdn]

            for abc_id in fqdn_abc_id.get(login, [None]):
                if not self._is_both_not_none(fqdn, abc_id):
                    continue

                group = self._get_or_create_group(abc_id)
                host = GroupResult.Host(fqdn, is_baremetal=False)
                group.hosts.append(host)

        self._load_services()
        for id_, service_obj in list(self._services.items()):
            self._groups_map[id_].name = self._qualify_name(service_obj.slug)


class ConductorGroups(GroupSourceBase):
    PREFIX = 'conductor'

    HOSTS_URL = urllib.parse.urljoin(settings.CONDUCTOR_URL, '/api/groups_export_cauth')
    RESPONSIBLES_URL = urllib.parse.urljoin(settings.CONDUCTOR_URL, '/api/groups/?format=json')

    @classmethod
    def _qualify_name(cls, name):
        name = name.replace('.', '_')
        return super(ConductorGroups, cls)._qualify_name(name)

    def load_hosts(self):
        data = HttpClient.fetch(self.HOSTS_URL, logger=self._logger, timeout=30).decode()
        for line in data.splitlines():
            line = line.strip()
            if not line:
                continue

            try:
                name, hosts, email, notify_email, st_queue = line.split(':', 4)
            except ValueError:
                continue

            name = self._qualify_name(name)
            item = GroupResult(
                name=name,
                source=self.PREFIX,
                contacts=GroupResult.Contacts(
                    request_email=email or None,
                    request_queue=st_queue.upper() or None,
                    notify_email=notify_email or None,
                ),
                hosts=[
                    GroupResult.Host(hostname=h.strip())
                    for h in hosts.split(',')
                    if should_be_added(h.strip())
                ],
            )
            if self._is_valid(item):
                self._groups_map[name] = item

    def load_responsibles(self):
        json_data = HttpClient.fetch(self.RESPONSIBLES_URL, logger=self._logger, timeout=60)
        try:
            data = json.loads(json_data)
        except Exception:
            self._logger.exception("Failed to decode json with responsibles")
            raise

        for group in data:
            name = self._qualify_name(group['name'])
            if name in self._groups_map:
                resps = {r.strip() for r in group['admins'] if r.strip()}
                self._groups_map[name].responsibles = resps


class GencfgGroupsBase(GroupSourceBase):
    def _get_resource_data(self, url):
        timeout = settings.CAUTH_SANDBOX_FETCH_TIMEOUT
        return HttpClient.fetch(url=url, timeout=timeout, logger=self._logger)

    @abc.abstractmethod
    def get_hosts_urls(self):
        return []

    @abc.abstractmethod
    def get_responsibles_urls(self):
        return []

    def load_hosts(self):
        for hosts_url in self.get_hosts_urls():
            self._load_hosts(hosts_url)

    def _load_hosts(self, url):
        data = self._get_resource_data(url)

        try:
            doc = lxml.etree.fromstring(data)
        except Exception:
            self._logger.exception("Failed to decode xml with hosts")
            raise

        if doc.tag != 'domain':
            self._logger.exception("Xml with hosts has wrong root element")

        for group in doc:
            if group.tag != 'group':
                continue

            name = self._qualify_name(group.attrib['name']).strip()
            hosts = {}  # typing: Dict[AnyStr, GroupResult.Host]
            for host in group:
                if host.tag != 'host':
                    continue

                is_dev = False
                for tag in host:
                    if tag.tag == 'type' and tag.text.strip() == 'dev-vm':
                        is_dev = True
                        break
                server_type = SERVER_TYPE.DEV_SERVER if is_dev else SERVER_TYPE.SERVER
                hostname = host.attrib['name'].strip()
                if hostname:
                    hosts[hostname] = GroupResult.Host(hostname=hostname, type=server_type)

            self._groups_map[name] = GroupResult(
                name=name,
                hosts=list(hosts.values()),
                source=self.PREFIX,
            )

    def load_responsibles(self):
        for responsibles_url in self.get_responsibles_urls():
            self._load_responsibles(responsibles_url)

    def _load_responsibles(self, url):
        data = self._get_resource_data(url)
        data = force_text(data)

        for line in data.splitlines():
            line = line.strip()
            if not line:
                continue

            try:
                name, responsibles = line.split('\t', 1)
            except ValueError:
                continue

            name = self._qualify_name(name)
            responsibles = {r.strip() for r in responsibles.split(',')
                            if r.strip()}

            if name in self._groups_map:
                self._groups_map[name].responsibles = responsibles


class GencfgGroups(GencfgGroupsBase):
    PREFIX = 'cms'

    SANDBOX_PROXY_URL = 'https://proxy.sandbox.yandex-team.ru/last/'

    def _get_resource_url(self, resource_type):
        q_params = urllib.parse.urlencode({'released': 'stable'})
        endpoint = '?'.join([resource_type, q_params])
        url = urllib.parse.urljoin(self.SANDBOX_PROXY_URL, endpoint)
        return url

    def get_hosts_urls(self):
        return [self._get_resource_url('GENCFG_CAUTH_EXPORT_HOSTS')]

    def get_responsibles_urls(self):
        return [self._get_resource_url('GENCFG_CAUTH_EXPORT_OWNERS')]


class YpGroups(GencfgGroupsBase):
    PREFIX = 'yp'

    YP_BASE_URL = 'https://yp-cauth-{datacenter}.n.yandex-team.ru/api/cauth/'
    DATA_CENTERS = settings.CAUTH_YP_DATA_CENTERS

    def _get_resource_urls(self, target):
        urls = []
        for datacenter in self.DATA_CENTERS:
            base_url = self.YP_BASE_URL.format(datacenter=datacenter)
            hosts_url = urllib.parse.urljoin(base_url, target)
            urls.append(hosts_url)
        return urls

    def get_hosts_urls(self):
        return self._get_resource_urls('domain')

    def get_responsibles_urls(self):
        return self._get_resource_urls('owners')


class WalleProjects(GroupSourceBase):
    PREFIX = 'walle'

    SOURCE_URL = urllib.parse.urljoin(settings.WALLE_URL, '/v1/cauth/projects-to-hosts')
    RESPONSIBLES_URL = urllib.parse.urljoin(settings.WALLE_URL, '/v1/cauth/projects-to-responsibles')
    SETTINGS_URL = urllib.parse.urljoin(settings.WALLE_URL, '/v1/cauth/projects-cauth-settings')

    def load_hosts(self):
        data = force_text(HttpClient.fetch(self.SOURCE_URL, logger=self._logger))
        for line in data.splitlines():
            line = line.strip()

            try:
                name, hosts = line.split(':')
                hosts = hosts.split(',') if hosts else []
            except ValueError:
                self._logger.exception("Malformed project-to-hosts mapping line: %s", line)
                raise

            name = self._qualify_name(name)
            self._groups_map[name] = GroupResult(
                name=name,
                source=self.PREFIX,
                hosts=[GroupResult.Host(hostname=h.strip()) for h in hosts],
            )

    def load_responsibles(self):
        json_data = HttpClient.fetch(self.RESPONSIBLES_URL, logger=self._logger, timeout=30)
        data = json.loads(json_data)

        for group, responsibles in data.items():
            name = self._qualify_name(group)
            if name in self._groups_map:
                self._groups_map[name].responsibles = set(responsibles)

    def load_settings(self):
        json_data = HttpClient.fetch(self.SETTINGS_URL, logger=self._logger, timeout=30)
        data = json.loads(json_data)

        for group, group_settings in data.items():
            name = self._qualify_name(group)
            if name in self._groups_map:
                trusted_sources = group_settings.get('trusted_sources', set())
                if trusted_sources:
                    trusted_sources = set(trusted_sources.split(','))

                self._groups_map[name].settings = GroupResult.Settings(
                    flow=group_settings.get('flow', None),
                    trusted_sources=trusted_sources,
                )

                key_sources = group_settings.get('key_sources', set())
                if key_sources:
                    key_sources = set(key_sources.split(','))

                self._groups_map[name].keys_info = GroupResult.KeysInfo(
                    key_sources=key_sources,
                    secure_ca_list_url=group_settings.get('secure_ca_list_url', None),
                    insecure_ca_list_url=group_settings.get('insecure_ca_list_url', None),
                    krl_url=group_settings.get('krl_url', None),
                    sudo_ca_list_url=group_settings.get('sudo_ca_list_url', None),
                )
