import abc
import datetime
import itertools
import contextlib
import collections
import logging

import attr
from django.conf import settings
from sqlalchemy import or_, and_
from typing import AnyStr, Dict, Set

from infra.cauth.server.common.alchemy import session_factory
from infra.cauth.server.common.constants import FLOW_TYPE, IDM_STATUS
from infra.cauth.server.common.models import (
    User,
    Source,
    Access,
    Server,
    ServerGroup,
    ServerResponsible,
    server_source_m2m,
)

from infra.cauth.server.master.api.idm.update import IdmUpdateAggregator
from infra.cauth.server.master.api.tasks import update_sources
from infra.cauth.server.master.importers.base import GeneralImporter
from infra.cauth.server.master.utils.config import get_class_path
from infra.cauth.server.master.utils.database import get_or_create
from infra.cauth.server.master.utils.dns_status import create_or_update_dns_status
from infra.cauth.server.master.utils.fqdn import should_be_pushed
from infra.cauth.server.master.utils.iterable import slice_iterator
from infra.cauth.server.master.utils.subtasks import SubtaskError
from functools import reduce


@attr.s
class ServerDesc(object):
    type = attr.ib(type=str)
    responsibles = attr.ib(type=Set)
    sources = attr.ib(type=Dict[AnyStr, Set])
    is_baremetal = attr.ib(type=bool, default=False)


ServerGroupDesc = collections.namedtuple('ServerGroupDesc', (
    'hosts',
    'responsibles',
    'request_email',
    'request_queue',
    'notify_email',
    'source',
    'flow',
    'trusted_sources',
    'key_sources',
    'secure_ca_list_url',
    'insecure_ca_list_url',
    'krl_url',
    'sudo_ca_list_url',
))


def get_source_map(session, source_names):
    sources = session.query(Source).filter(Source.name.in_(source_names)).all()

    if len(sources) != len(source_names):
        invalid_sources = set(source_names) - {source.name for source in sources}
        raise RuntimeError('Invalid sources: {}'.format(invalid_sources))

    return {source.name: source for source in sources}


def get_user_map(session, logins):
    if logins:
        user_query = session.query(User).filter(User.login.in_(logins))
        return {u.login: u for u in user_query}
    else:
        return {}


class BaseImportSubtask(object, metaclass=abc.ABCMeta):
    time_limit = 120 * 60

    def __init__(self, data, suite_run_id):
        self.data = data
        self.suite_run_id = suite_run_id
        self.idm_update = IdmUpdateAggregator(self.suite_run_id)
        self.logger = logging.getLogger(get_class_path(self))

    @abc.abstractmethod
    def do_one(self, item):
        pass

    def run(self):
        self.logger.info(
            'starting {s.__class__.__name__} (suite_run_id={s.suite_run_id})'.format(s=self)
        )

        for item in self.data:
            self.do_one(item)

        self.idm_update.commit()


class DstRemoveSubtask(BaseImportSubtask):
    def do_one(self, item):
        self.logger.info('Remove {}'.format(item))
        self.idm_update.add_destination(item)


class ServerUpdateSubtask(BaseImportSubtask):
    @staticmethod
    def _deserialize_server(item):
        return ServerDesc(
            type=item['type'],
            responsibles={src: set(logins) for src, logins in list(item['responsibles'].items())},
            sources=set(item['sources']),
            is_baremetal=item['is_baremetal'],
        )

    def do_one(self, item):
        fqdn, item = item
        desc = self._deserialize_server(item)
        with contextlib.closing(session_factory()) as session:
            source_map = get_source_map(session, desc.sources)
            server, is_created = get_or_create(session, Server, 'fqdn', fqdn.lower())
            create_or_update_dns_status(session, server)

            changes = {
                'created_server': is_created,
                'type': {
                    'previous': server.type,
                },
                'sources': {
                    'previous': set(server.sources),
                },
                'responsibles': {
                    'previous': set(server.responsibles),
                },
            }

            server.type = desc.type
            server.is_baremetal = desc.is_baremetal
            changes['type']['new'] = server.type

            responsible_logins = reduce(lambda a, b: a | set(b),
                                        list(desc.responsibles.values()), set())
            user_map = get_user_map(session, responsible_logins)

            sticky_sources = [
                source for source in server.sources
                if source.name in settings.CAUTH_STICKY_SOURCES and source.name not in source_map.keys()
            ]

            server.sources = list(source_map.values()) + sticky_sources
            changes['sources']['new'] = set(server.sources)

            self._update_responsibles(session, desc.responsibles, server, source_map, user_map)
            changes['responsibles']['new'] = set(server.responsibles)

            resp_changes = changes['responsibles']
            if should_be_pushed(server.fqdn) and (is_created or resp_changes['previous'] != resp_changes['new']):
                if server.idm_status == IDM_STATUS.ACTUAL:
                    server.idm_status = IDM_STATUS.DIRTY
                    server.became_dirty_at = datetime.datetime.now()
                if (
                    server.first_pending_push_started_at is None
                    or (
                        server.last_push_ended_at
                        and server.last_push_ended_at > server.first_pending_push_started_at
                    )
                ):
                    server.first_pending_push_started_at = datetime.datetime.now()
                self.idm_update.add_destination(fqdn)

            self._log_update(server, changes)
            session.commit()

    @staticmethod
    def _update_responsibles(session, responsible_map, server, source_map, user_map):
        keep_responsibles = set()

        for src_name, logins in list(responsible_map.items()):
            for login in logins:
                keep_responsibles.add((src_name, login))

        current_responsibles = set()
        for resp in list(server.responsibles):
            key = (resp.source.name, resp.user.login)
            if key in keep_responsibles:
                current_responsibles.add(key)
            else:
                session.delete(resp)
                server.responsibles.remove(resp)

        for src_name, login in keep_responsibles:
            key = (src_name, login)
            if key not in current_responsibles and login in user_map:
                resp = ServerResponsible(
                    server=server,
                    user=user_map[login],
                    source=source_map[src_name],
                )
                session.add(resp)

    @staticmethod
    def _format_resp(resp):
        return '{0.user.login} ({0.source.name})'.format(resp)

    def _log_update(self, server, changes):
        if changes['created_server']:
            formatted_resps = list(map(self._format_resp, server.responsibles))
            formatted_sources = [src.name for src in server.sources]

            self.logger.info('Adding new_server %s type: %s responsibles: {%s} '
                             'sources: {%s}',
                             server.fqdn,
                             server.type,
                             ', '.join(formatted_resps),
                             ', '.join(formatted_sources))
        else:
            changed_resps = []
            changed_sources = []

            resps = changes['responsibles']
            sources = changes['sources']
            removed_resps = resps['previous'] - resps['new']
            added_resps = resps['new'] - resps['previous']
            removed_sources = sources['previous'] - sources['new']
            added_sources = sources['new'] - sources['previous']

            for resp in removed_resps:
                changed_resps.append('-{}'.format(self._format_resp(resp)))

            for resp in added_resps:
                changed_resps.append('+{}'.format(self._format_resp(resp)))

            for src in removed_sources:
                changed_sources.append('-{}'.format(src.name))

            for src in added_sources:
                changed_sources.append('+{}'.format(src.name))

            old_type = changes['type']['previous']
            new_type = changes['type']['new']

            if changed_resps or changed_sources or old_type != new_type:
                parts = []
                if old_type != new_type:
                    parts.append('type: "{}"->"{}"'.format(old_type, new_type))
                if changed_resps:
                    items = ', '.join(changed_resps)
                    parts.append('responsibles: {%s}' % items)
                if changed_sources:
                    items = ', '.join(changed_sources)
                    parts.append('sources: {%s}' % items)

                self.logger.info('Updating new_server %s %s', server.fqdn, ' '.join(parts))


class ServerGroupUpdateSubtask(BaseImportSubtask):
    def do_one(self, item):
        """ Если группа создана, то мы должны пропушить ее в IDM, то же самое если у нее изменились
        ответственные, мыло или список хостов (новая логика - список хостов влияет на
        ответственность в IDM). Если группа была создана, изменились ответственные или мыло то мы
        должны пропушить в IDM ВСЕ хосты группы. Если изменился только список хостов то пропушивать
        все хосты не нужно, так как если после добавления хоста группа стала не валидной
        (конкуренция доверенных источников) то ответственность за хосты не изменится, так как хосты
        и так выгружают ответственных только групп доверенных источников.
        """
        name, item = item
        desc = ServerGroupDesc(**item)
        with contextlib.closing(session_factory()) as session:
            source_map = get_source_map(session, [desc.source])
            source = source_map[desc.source]

            # Во время обновления эксклюзивно лочим строку в new_server_groups
            group, is_created = get_or_create(session, ServerGroup, 'name', name, for_update=True)

            responsible_users = list(get_user_map(session, desc.responsibles).values())
            hostnames = [host['hostname'] for host in desc.hosts]
            servers = list(session.query(Server).filter(Server.fqdn.in_(hostnames))) if hostnames else []

            server_by_fqdn = {server.fqdn: server for server in itertools.chain(servers, group.servers)}

            changes = {
                'created_group': is_created,
                'request_email': {
                    'previous': group.email,
                    'new': desc.request_email,
                },
                'request_queue': {
                    'previous': group.request_queue,
                    'new': desc.request_queue,
                },
                'notify_email': {
                    'previous': group.notify_email,
                    'new': desc.notify_email,
                },
                'source': {
                    'previous': group.source.name if group.source else None,
                    'new': desc.source,
                },
                'hosts': {
                    'previous': {s.fqdn for s in group.servers},
                    'new': {s.fqdn for s in servers},
                },
                'responsibles': {
                    'previous': {u.login for u in group.responsible_users},
                    'new': {u.login for u in responsible_users},
                },
                'flow': {
                    'previous': group.flow,
                    'new': desc.flow or FLOW_TYPE.CLASSIC,
                },
                'trusted_sources': {
                    'previous': {s.name for s in group.trusted_sources},
                    'new': set(desc.trusted_sources),
                },
                'key_sources': {
                    'previous': {s for s in group.key_sources.split(',')} if group.key_sources else set(),
                    'new': set(desc.key_sources),
                },
                'secure_ca_list_url': {
                    'previous': group.secure_ca_list_url,
                    'new': desc.secure_ca_list_url,
                },
                'insecure_ca_list_url': {
                    'previous': group.insecure_ca_list_url,
                    'new': desc.insecure_ca_list_url,
                },
                'krl_url': {
                    'previous': group.krl_url,
                    'new': desc.krl_url,
                },
                'sudo_ca_list_url': {
                    'previous': group.sudo_ca_list_url,
                    'new': desc.sudo_ca_list_url,
                },
            }

            host_changes = changes['hosts']
            resp_changes = changes['responsibles']
            trusted_sources_changes = changes['trusted_sources']
            key_sources_changes = changes['key_sources']

            is_resps_changed = resp_changes['previous'] != resp_changes['new']
            is_email_changed = group.email != desc.request_email

            if is_created or is_resps_changed or is_email_changed:
                if group.idm_status == IDM_STATUS.ACTUAL:
                    group.idm_status = IDM_STATUS.DIRTY
                    group.became_dirty_at = datetime.datetime.now()
                if (
                    group.first_pending_push_started_at is None
                    or (
                        group.last_push_ended_at
                        and group.last_push_ended_at > group.first_pending_push_started_at
                    )
                ):
                    group.first_pending_push_started_at = datetime.datetime.now()
                self.idm_update.add_destination(name)

                for fqdn in host_changes['previous'] & host_changes['new']:
                    if not should_be_pushed(fqdn):
                        continue
                    server = server_by_fqdn[fqdn]
                    if server.idm_status == IDM_STATUS.ACTUAL:
                        server.idm_status = IDM_STATUS.DIRTY
                        server.became_dirty_at = datetime.datetime.now()
                    if (
                        server.first_pending_push_started_at is None
                        or (
                            server.last_push_ended_at
                            and server.last_push_ended_at > server.first_pending_push_started_at
                        )
                    ):
                        server.first_pending_push_started_at = datetime.datetime.now()
                    self.idm_update.add_destination(fqdn)

            for fqdn in host_changes['previous'] ^ host_changes['new']:
                if not should_be_pushed(fqdn):
                    continue
                server = server_by_fqdn[fqdn]
                if server.idm_status == IDM_STATUS.ACTUAL:
                    server.idm_status = IDM_STATUS.DIRTY
                    server.became_dirty_at = datetime.datetime.now()
                if (
                    server.first_pending_push_started_at is None
                    or (
                        server.last_push_ended_at
                        and server.last_push_ended_at > server.first_pending_push_started_at
                    )
                ):
                    server.first_pending_push_started_at = datetime.datetime.now()
                self.idm_update.add_destination(fqdn)

            group.email = desc.request_email
            group.request_queue = desc.request_queue
            group.notify_email = desc.notify_email
            group.source = source
            group.responsible_users = responsible_users
            group.servers = servers
            if group.source.is_modern:
                if desc.flow:
                    group.flow = desc.flow
                if desc.trusted_sources and (
                        trusted_sources_changes['previous'] != trusted_sources_changes['new']
                ):
                    update_sources(session, group, desc.trusted_sources, do_commit=False)

            if key_sources_changes['previous'] != key_sources_changes['new']:
                group.key_sources = ','.join(sorted(desc.key_sources))
            group.secure_ca_list_url = desc.secure_ca_list_url
            group.insecure_ca_list_url = desc.insecure_ca_list_url
            group.krl_url = desc.krl_url
            group.sudo_ca_list_url = desc.sudo_ca_list_url

            self._log_update(group, changes)

            session.commit()

    def _log_update(self, group, changes):
        if changes['created_group']:
            self.logger.info(
                'Adding group %s source=%s email=%s hosts: {%s} '
                'responsibles: {%s} flow=%s trusted_sources: {%s} '
                'key_sources: {%s}',
                group.name, group.source.name, group.email,
                ', '.join(changes['hosts']['new']),
                ', '.join(changes['responsibles']['new']),
                group.flow,
                ', '.join(changes['trusted_sources']['new']),
                group.key_sources
            )
        else:
            changed_resps = []
            changed_hosts = []
            changed_attrs = []
            changed_trusted_sources = []
            changed_key_sources = []

            resps = changes['responsibles']
            removed_resps = resps['previous'] - resps['new']
            added_resps = resps['new'] - resps['previous']

            hosts = changes['hosts']
            removed_hosts = hosts['previous'] - hosts['new']
            added_hosts = hosts['new'] - hosts['previous']

            trusted_sources = changes['trusted_sources']
            removed_trusted_sources = trusted_sources['previous'] - trusted_sources['new']
            added_trusted_sources = trusted_sources['new'] - trusted_sources['previous']

            key_sources = changes['key_sources']
            removed_key_sources = key_sources['previous'] - key_sources['new']
            added_key_sources = key_sources['new'] - key_sources['previous']

            for attr_name in (
                    'source', 'flow', 'request_email', 'request_queue', 'notify_email',
                    'secure_ca_list_url', 'insecure_ca_list_url', 'krl_url', 'sudo_ca_list_url',
            ):
                if changes[attr_name]['previous'] != changes[attr_name]['new']:
                    changed_attrs.append('{}: {} -> {}'.format(
                        attr_name,
                        changes[attr_name]['previous'],
                        changes[attr_name]['new']))

            for resp in removed_resps:
                changed_resps.append('-{}'.format(resp))

            for resp in added_resps:
                changed_resps.append('+{}'.format(resp))

            for host in removed_hosts:
                changed_hosts.append('-{}'.format(host))

            for host in added_hosts:
                changed_hosts.append('+{}'.format(host))

            for trusted_source in removed_trusted_sources:
                changed_trusted_sources.append('-{}'.format(trusted_source))

            for trusted_source in added_trusted_sources:
                changed_trusted_sources.append('+{}'.format(trusted_source))

            for key_source in removed_key_sources:
                changed_key_sources.append('-{}'.format(key_source))

            for key_source in added_key_sources:
                changed_key_sources.append('+{}'.format(key_source))

            if changed_attrs or changed_resps or changed_hosts or changed_trusted_sources or changed_key_sources:
                parts = list(changed_attrs)

                if changed_resps:
                    items = ', '.join(changed_resps)
                    parts.append('responsibles: {%s}' % items)

                if changed_hosts:
                    items = ', '.join(changed_hosts)
                    parts.append('hosts: {%s}' % items)

                if changed_trusted_sources:
                    items = ', '.join(changed_trusted_sources)
                    parts.append('trusted_sources: {%s}' % items)

                if changed_key_sources:
                    items = ', '.join(changed_key_sources)
                    parts.append('key_sources: {%s}' % items)

                self.logger.info('Updating group %s %s', group.name, ' '.join(parts))


class BaseServersDatabaseImporter(GeneralImporter, metaclass=abc.ABCMeta):
    TARGET = 'database'
    sanity_limits = {'any': 5000}

    REMOVE_SLICE_SIZE = 5000
    UPDATE_SLICE_COUNT = None

    model = None
    model_slug = 'id'

    @abc.abstractmethod
    def serialize_for_subtask(self, desc):
        pass

    @abc.abstractmethod
    def get_query_for_delete(self, session, sticky_sources, delete_ids):
        return []

    def run_import(self):
        self.logger.info(
            'Starting importer {s.__class__.__name__} (suite_run_id={s.suite.run_id})'
            .format(s=self)
        )

        parsed = self.validate_new_data()
        data = self.load_new_data(parsed)

        removed_slugs = self.remove_extra(data)

        for items in slice_iterator(removed_slugs, self.REMOVE_SLICE_SIZE):
            self.add_subtask(
                name='remove',
                data=list(items),
            )

        for items in slice_iterator(iter(data.items()), self.UPDATE_SLICE_COUNT):
            self.add_subtask(
                name='update',
                data=[(slug, self.serialize_for_subtask(desc)) for slug, desc in items],
            )

        try:
            self.subtask_pool.join()
        except SubtaskError as error:
            for task in error.tasks:
                self.logger.error('Subtask {task[task_id]} failed: {task[error]}'.format(task=task))
            raise

        self.logger.info('All subtasks finished')

    def remove_extra(self, data):
        with contextlib.closing(session_factory()) as session:
            self.logger.info('Reading all existing slugs from database')
            query = session.query(self.model.id, getattr(self.model, self.model_slug))
            existing_slugs2ids = {slug: object_id for object_id, slug in query}

            fresh_time = datetime.datetime.now() - datetime.timedelta(
                hours=settings.CAUTH_SKIP_REMOVING_ON_IMPORT_HOURS
            )
            fresh_slugs = {
                slug for (slug,) in session.query(
                    getattr(self.model, self.model_slug)
                ).filter(self.model.created_at > fresh_time)
            }

            default_source = session.query(Source).filter_by(is_default=True).first()
            sticky_sources = session.query(Source).filter(Source.name.in_(settings.CAUTH_STICKY_SOURCES))
            sticky_sources = list(sticky_sources) + [default_source]

            if not default_source:
                raise RuntimeError('Default source does not exist')
            i_delete_slugs = existing_slugs2ids.keys() - data.keys() - fresh_slugs

            deleted_slugs = []
            for items in slice_iterator(i_delete_slugs, 1000):
                delete_ids = [existing_slugs2ids[slug] for slug in items]

                query = self.get_query_for_delete(session, sticky_sources, delete_ids)

                delete_ids = []
                for object_id, slug in query:
                    self.logger.info('Removing {} {}'.format(self.model.__name__, slug))
                    delete_ids.append(object_id)
                    deleted_slugs.append(slug)

                if not delete_ids:
                    continue

                delete_query = session.query(self.model).filter(self.model.id.in_(delete_ids))
                delete_query.delete(synchronize_session=False)
                session.commit()

            return deleted_slugs


class NewServerImporter(BaseServersDatabaseImporter):
    __item_attrs__ = list(attr.fields_dict(ServerDesc))

    UPDATE_SLICE_COUNT = 5000

    model = Server
    model_slug = 'fqdn'

    subtask_classes = {
        'remove': DstRemoveSubtask,
        'update': ServerUpdateSubtask,
    }

    def get_query_for_delete(self, session, sticky_sources, delete_ids):
        """ отфильтровываем сервера, у которых нет дефолтного или "липкого"
        источника, и берём на них локи.
        таким образом мы не удалим записи, добавленные после
        первоначального селекта (они в этот момент могут быть
        добавлены только ручкой /add_server/ с липким источником)
        """

        return (
            session.query(Server.id, Server.fqdn)
            .join(
                server_source_m2m,
                and_(
                    server_source_m2m.c.server_id == Server.id,
                    server_source_m2m.c.source_id.notin_([source.id for source in sticky_sources])
                ),
            )
            .filter(
                Server.id.in_(delete_ids),
            )
            .with_for_update()
        )

    @staticmethod
    def _flatten_server_groups(groups):
        for grp in groups:
            for host in grp['hosts']:
                yield {
                    'name': host['hostname'],
                    'type': host['type'],
                    'responsibles': {grp['source']: set(grp['responsibles'])},
                    'sources': {grp['source']},
                    'is_baremetal': host['is_baremetal'],
                }

    @staticmethod
    def _server_with_sources(server):
        return {
            'name': server['name'],
            'type': server['type'],
            'responsibles': server['responsibles'],
            'sources': set(server['responsibles'].keys()),
            'is_baremetal': server['is_baremetal'],
        }

    def serialize_for_subtask(self, desc):
        return {
            'type': desc.type,
            'responsibles': {src: list(logins) for src, logins in list(desc.responsibles.items())},
            'sources': list(desc.sources),
            'is_baremetal': desc.is_baremetal,
        }

    def load_new_data(self, input_data):
        data = {}
        chain = itertools.chain(
            self._flatten_server_groups(input_data['groups']),
            map(self._server_with_sources, input_data['servers']),
        )

        for srv in chain:
            key = srv['name'].lower()
            desc = data.setdefault(key, ServerDesc(
                type=srv['type'],  # тип для хоста определяется по первому вхождению
                responsibles=collections.defaultdict(set),
                sources=set(),
            ))
            if srv['is_baremetal']:
                desc.is_baremetal = True

            for src in srv['sources']:
                desc.sources.add(src)

            for src, responsibles in list(srv['responsibles'].items()):
                for login in responsibles:
                    desc.responsibles[src].add(login)

        return data


class NewServerGroupImporter(BaseServersDatabaseImporter):
    __item_attrs__ = ServerGroupDesc._fields

    UPDATE_SLICE_COUNT = 500

    model = ServerGroup
    model_slug = 'name'

    subtask_classes = {
        'remove': DstRemoveSubtask,
        'update': ServerGroupUpdateSubtask,
    }

    def get_query_for_delete(self, session, sticky_sources, delete_ids):
        return (
            session.query(ServerGroup.id, ServerGroup.name)
            .filter(
                ServerGroup.id.in_(delete_ids),
                ServerGroup.source_id.notin_([source.id for source in sticky_sources])
            )
            .with_for_update()
        )

    def serialize_for_subtask(self, desc):
        return dict(desc._asdict())

    def run_import(self):
        super(NewServerGroupImporter, self).run_import()

        self._update_access_rename_dst()
        self._update_access_restore_dst_ids()

    def load_new_data(self, input_data):
        data = {}
        for grp in input_data['groups']:
            data[grp['name']] = ServerGroupDesc(
                responsibles=grp['responsibles'],
                hosts=grp['hosts'],
                request_email=grp['contacts']['request_email'],
                request_queue=grp['contacts']['request_queue'],
                notify_email=grp['contacts']['notify_email'],
                source=grp['source'],
                flow=grp['settings']['flow'],
                trusted_sources=grp['settings']['trusted_sources'],
                key_sources=grp['keys_info']['key_sources'],
                secure_ca_list_url=grp['keys_info']['secure_ca_list_url'],
                insecure_ca_list_url=grp['keys_info']['insecure_ca_list_url'],
                krl_url=grp['keys_info']['krl_url'],
                sudo_ca_list_url=grp['keys_info']['sudo_ca_list_url'],
            )

        return data

    @staticmethod
    def _update_access_rename_dst():
        with contextlib.closing(session_factory()) as session:
            server_query = (
                session.query(Access, Server.fqdn)
                .join(Access.dst_server)
                .filter(Access.dst != Server.fqdn)
            )

            for rule, new_fqdn in server_query:
                rule.dst = new_fqdn

            group_query = (
                session.query(Access, ServerGroup.name)
                .join(Access.dst_group)
                .filter(Access.dst != ServerGroup.name)
            )

            for rule, new_name in group_query:
                rule.dst = new_name

            session.commit()

    @staticmethod
    def _update_access_restore_dst_ids():
        with contextlib.closing(session_factory()) as session:
            query = (
                session.query(Access, Server.id, ServerGroup.id)
                .outerjoin(Server, Server.fqdn == Access.dst)
                .outerjoin(ServerGroup, ServerGroup.name == Access.dst)
                .filter(
                    Access.dst_server_id.is_(None),
                    Access.dst_group_id.is_(None),
                    or_(
                        Server.id.isnot(None),
                        ServerGroup.id.isnot(None),
                    )
                )
            )

            for rule, server_id, group_id in query:
                rule.dst_server_id = server_id
                rule.dst_group_id = group_id

            session.commit()
