import json
import urllib.request
import urllib.parse
import urllib.error
from marshmallow import ValidationError

from django.conf import settings

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import SkippedStaffInconsistencies
from infra.cauth.server.master.utils.http_client import HttpClient
from infra.cauth.server.master.api.models import EnabledWikiGroup
from infra.cauth.server.master.importers.staff import validators, transformations

PAGE_SIZE = 1000


def call_staff_api_v3(entity, logger, params=None, url=None):
    if not url:
        url = 'https://{host}/v3/{entity}?{query}'.format(
            host=settings.STAFF_API_HOST,
            entity=entity,
            query=urllib.parse.urlencode(params) if params else '',
        )

    response = HttpClient.fetch(url, oauth=True, logger=logger)

    try:
        response = response.decode('utf-8')
    except UnicodeDecodeError:
        logger.error('Failed to decode response: %s', repr(response))
        raise

    try:
        return json.loads(response)
    except ValueError:
        logger.error('Failed to parse response as json: %s', repr(response))
        raise


def _fetch_entity(entity, params, logger):
    url = None
    result = []

    params['_limit'] = str(PAGE_SIZE)

    while True:
        response = call_staff_api_v3(entity=entity, url=url, params=params,
                                     logger=logger)
        result += response['result']

        if 'links' in response and 'next' in response['links']:
            url = response['links']['next']
            params = None
        else:
            return result


def _check_skipping_sanity(items_name, valid_count, invalid_count, ratio_limit):
    """
    Returns [] if invalids ratio is sane, [error_message] otherwise
    """
    if invalid_count == 0:
        return []
    total = valid_count + invalid_count
    invalid_ratio = float(invalid_count) / total
    if invalid_ratio > ratio_limit:
        return ['Too much {items_name} to skip: {invalid_count}/{total} > {limit};'.format(
            items_name=items_name,
            invalid_count=invalid_count,
            total=total,
            limit=ratio_limit,
        )]
    return []


def fetch_staff(logger):
    types = ['department', 'wiki', 'service', 'servicerole']
    groups = _fetch_entity(
        entity='groups',
        params={
            'type': ','.join(types),
            'is_deleted': 'false',
            '_sort': 'id',
            '_fields': ','.join((
                'id',
                'type',
                'url',
                'service.id',
                'department.id',
                'ancestors.id',
                'ancestors.department.id',
                'ancestors.is_deleted',
                'parent.id',
                'parent.service.id',
                'parent.department.id',
            )),
        },
        logger=logger,
    )

    persons = _fetch_entity(
        entity='persons',
        params={
            '_sort': 'login',
            '_fields': ','.join((
                'id',
                'login',
                'name.first.en',
                'name.last.en',
                'official.is_dismissed',
                'official.is_robot',
                'official.join_at',
                'environment.shell',
                'keys.key',
                'department_group.department.id',
                'department_group.id',
                'groups.group.id',
                'groups.group.type',
                'groups.group.is_deleted',
            )),
        },
        logger=logger,
    )

    persons_format_errors = []
    try:
        valid_persons = validators.PersonFetchValidator(many=True).load(persons)
    except (ValidationError, ValueError) as e:
        broken_data = e.messages
        logger.exception('Invalid persons from staff: %s', broken_data)
        if not settings.SKIP_BROKEN_STAFF_DATA:
            raise

        persons_format_errors = [
            '{}: {}'.format(
                item['person'].get('login'),
                json.dumps(item['error'], sort_keys=True)
            ) for item in broken_data
        ]
        valid_persons = e.valid_data

    groups_format_errors = []

    try:
        valid_groups = validators.GroupFetchValidator(many=True).load(groups)
    except (ValidationError, ValueError) as e:
        broken_data = e.messages
        logger.exception('Invalid groups from staff: %s', broken_data)
        if not settings.SKIP_BROKEN_STAFF_DATA:
            raise

        groups_format_errors = [
            '{}: {}'.format(
                item['group'].get('url'),
                json.dumps(item['error'], sort_keys=True)
            ) for item in broken_data
        ]
        valid_groups = e.valid_data

    try:
        validated = validators.StaffStructureValidator().dump({
            'persons': valid_persons,
            'groups': valid_groups,
        })
        valid_persons = validated['persons']
        valid_groups = validated['groups']

        if settings.SKIP_BROKEN_STAFF_DATA:
            groups_consistency_errors = validated['groups_errors']
            persons_consistency_errors = validated['persons_errors']
            groups_errors = groups_format_errors + groups_consistency_errors
            persons_errors = persons_format_errors + persons_consistency_errors
            sanity_checks = (
                _check_skipping_sanity('groups', len(valid_groups), len(groups_errors), settings.GROUP_ERRORS_SKIP_LIMIT_RATIO)
                + _check_skipping_sanity('profiles', len(valid_persons), len(persons_errors), settings.PROFILE_ERRORS_SKIP_LIMIT_RATIO)
            )

            errors = groups_errors + persons_errors
            if sanity_checks:
                raise ValidationError("Sanity checks failed during skipping staff inconsistencies: '{checks}'. Inconsistencies are: {errors}".format(
                    checks=' '.join(sanity_checks),
                    errors=validators._errors_to_str(errors)
                ))

            # truncate inconsistencies for monitoring
            SkippedStaffInconsistencies.query.delete()
            # add to monitoring
            Session.add_all(SkippedStaffInconsistencies(error=str(error)) for error in errors)
            Session.commit()
    except ValidationError as e:
        logger.exception('Inconsistent data from staff: {}'.format(e))
        raise

    transformations.create_memberships(valid_persons, valid_groups)
    valid_groups = list(map(transformations.convert_group, valid_groups))
    valid_persons = list(map(transformations.convert_person, valid_persons))

    enabled_wiki_groups = {g.name for g in EnabledWikiGroup.query.all()}
    is_dpt_or_svc = lambda g: g['type'] in ('dpt', 'svc', 'svcrole')
    is_enabled_wiki = lambda g: g['type'] == 'wiki' and g['name'] in enabled_wiki_groups

    return {
        'persons': valid_persons,
        'groups': [g for g in valid_groups if is_dpt_or_svc(g) or is_enabled_wiki(g)],
    }
