import subprocess
import tempfile
import collections

from django.conf import settings

from ..base import ImportSuite
from ..registry import SuiteGroup
from . import to_database, to_ldap


class StaffDatabaseSuite(ImportSuite):
    TARGET = 'database'
    IMPORTER_CLASSES = (
        to_database.NewGroupsImporter,
        to_database.NewUserImporter,
        to_database.NewPublicKeyImporter,
        to_database.NewGroupMembersImporter,
    )


class StaffLdapSuite(ImportSuite):
    TARGET = 'ldap'
    IMPORTER_CLASSES = (
        to_ldap.PeopleLdapImporter,
        to_ldap.GroupsLdapImporter,
    )

    def __init__(self, *args, **kwargs):
        super(StaffLdapSuite, self).__init__(*args, **kwargs)
        self.out_stream = tempfile.TemporaryFile(mode='w')

    def post_run(self, *args, **kwargs):
        self.out_stream.seek(0)

        ldapmodify = subprocess.Popen(
            args=('/usr/bin/ldapmodify', '-h', settings.LDAP_SERVER,
                  '-D', settings.LDAP_USER, '-w', settings.LDAP_PASSWORD),
            stdin=self.out_stream,
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
        )

        stdout, stderr = ldapmodify.communicate()

        for line in stdout.decode().splitlines():
            if line:
                self.logger.info("ldapmodify stdout: %s", line)

        for line in stderr.decode().splitlines():
            if line:
                self.logger.warning("ldapmodify stderr: %s", line)

        if ldapmodify.returncode:
            raise RuntimeError("ldapmodify returned a nonzero exit code: %s",
                               ldapmodify.returncode)

        super(StaffLdapSuite, self).post_run(*args, **kwargs)


SuiteGroup(
    name='staff',
    fetcher='.staff.retrievers.fetch_staff',
    validator='.staff.validators.StaffImportValidator',
    suites=collections.OrderedDict((
        ('database', StaffDatabaseSuite),
        ('ldap', StaffLdapSuite),
    )),
)
