import collections
import itertools
import logging

from django.conf import settings


BASE_GROUP_OFFSET = 23000
BASE_LEGACY_OFFSET = 20000
LEGACY_OVERLAP = BASE_GROUP_OFFSET - BASE_LEGACY_OFFSET

logger = logging.getLogger(__name__)


def make_dpt_gid(department_id, group_id):
    if department_id < LEGACY_OVERLAP:
        return BASE_LEGACY_OFFSET + department_id
    else:
        return BASE_GROUP_OFFSET + group_id


def convert_group(group):
    ancestors = []
    parent = None
    service_id = None
    gid = BASE_GROUP_OFFSET + group['id']
    type_ = group['type']

    if type_ == 'department':
        name = '_'.join(('dpt', group['url']))
        gid = make_dpt_gid(group['department']['id'], group['id'])
        ancestors = [make_dpt_gid(a['department']['id'], a['id'])
                     for a in group['ancestors']]
        type_ = 'dpt'

        if 'parent' in group:
            parent = make_dpt_gid(group['parent']['department']['id'],
                                  group['parent']['id'])

    elif type_ == 'wiki':
        name = '_'.join(('wiki', group['url']))
    elif type_ == 'service':
        name = group['url']
        type_ = 'svc'
        service_id = group['service']['id']
    elif type_ == 'servicerole':
        name = group['url']
        type_ = 'svcrole'
        service_id = group['parent']['service']['id']
    else:
        raise ValueError

    return {
        'gid': gid,
        'name': name,
        'parent_gid': parent,
        'members': group['members'],
        'ancestors': ancestors,
        'type': type_,
        'service_id': service_id,
        'staff_id': group['id'],
    }


class _UnknownCommonGroup(Exception):
    """Can't find common group for person"""


def convert_person(person):
    fired = person['official']['is_dismissed']

    if fired:
        department_id = 0
        group_id = 0
    else:
        department_id = person['department_group']['department']['id']
        group_id = person['department_group']['id']

    first_name = person['name']['first']['en']
    last_name = person['name']['last']['en']

    return {
        'login': person['login'],
        'uid': person['id'] + BASE_LEGACY_OFFSET,
        'gid': make_dpt_gid(department_id, group_id),
        'is_fired': fired,
        'is_robot': person['official']['is_robot'],
        'join_date': person['official']['join_at'],
        'name': ' '.join((first_name, last_name)).strip(),
        'first_name': first_name,
        'last_name': last_name,
        'shell': person['environment']['shell'],
        'pub_keys': [key['key'] for key in person.get('keys', [])],
    }


def create_memberships(raw_persons, raw_groups):
    department_groups_map = {}
    common_groups_map = {}

    common_group_types = ['wiki', 'service', 'servicerole']
    for group in raw_groups:
        group['members'] = []
        if group['type'] == 'department':
            department_groups_map[group['department']['id']] = group
        elif group['type'] in common_group_types:
            common_groups_map[group['id']] = group

    skipped_persons = set()

    for person in raw_persons:
        if person['official']['is_dismissed']:
            continue

        dpt_id = person['department_group']['department']['id']
        if dpt_id not in department_groups_map:
            if settings.SKIP_BROKEN_STAFF_DATA:
                logger.warning("Department group '%s' for user '%s' does not exist, so skip", dpt_id, person['login'])
                skipped_persons.add(person['login'])
                continue
            raise RuntimeError("Department group for user %s does not exist" % person['login'])

        common_groups_update = collections.defaultdict(list)
        try:
            for group in person.get('groups', []):
                group_value = group.get('group', {})

                if group_value.get('is_deleted'):
                    continue

                if group_value.get('type') in common_group_types:
                    grp_id = group_value.get('id')
                    if grp_id not in common_groups_map:
                        if settings.SKIP_BROKEN_STAFF_DATA:
                            logger.warning("Common group '%s' for user '%s' does not exist, so skip", grp_id,
                                           person['login'])
                            raise _UnknownCommonGroup()
                        raise RuntimeError("Common group %s for user %s does not exist" % (grp_id, person['login']))

                    common_groups_update[grp_id].append(person['login'])
        except _UnknownCommonGroup:
            skipped_persons.add(person['login'])
            continue

        dpt_group = department_groups_map[dpt_id]
        dpt_group['members'].append(person['login'])

        for grp_id, logins in common_groups_update.items():
            common_group = common_groups_map[grp_id]
            common_group['members'].extend(logins)

    if len(skipped_persons) > len(raw_persons) * settings.PROFILE_ERRORS_SKIP_LIMIT_RATIO:
        raise RuntimeError(
            "Too many skipped persons: {}, ...".format(", ".join(itertools.islice(skipped_persons, 3)))
        )
