from django.conf import settings

from marshmallow import Schema, decorators, fields as f, validate as v, ValidationError
from infra.cauth.server.master.utils.validation import CauthOneOfSchema


class ConsistencyError(Exception):
    def __init__(self, message):
        self.message = message

    def __str__(self):
        return self.message


class GroupReferenceError(Exception):
    def __init__(self, reference, message):
        self.reference = reference
        self.message = message

    def __str__(self):
        return "{0.message} (referenced in {0.reference})".format(self)


def _check_group_reference(group, ref_entity, group_map):
    if group.get('is_deleted'):
        return

    if group['id'] not in group_map:
        raise GroupReferenceError(
            reference=ref_entity,
            message="Group with id={} not found".format(group['id']),
        )

    if group.get('type') == 'department' and 'department' in group:
        if group_map[group['id']] != group['department']['id']:
            raise GroupReferenceError(
                reference=ref_entity,
                message="Group with id={} found, but referencing "
                        "department.id does not match".format(group['id']),
            )


def generate_group_map(groups):
    group_map = {}
    for group in groups:
        department_id = None
        if group['type'] == 'department':
            department_id = group['department']['id']

        group_map[group['id']] = department_id

    return group_map


def _check_group(group, group_map):
    errors = []
    ref_entity = '<Group: {0[id]}, {0[type]}, {0[url]}>'.format(group)

    dependent_groups = group.get('ancestors', [])
    if group.get('parent'):
        dependent_groups.append(group['parent'])

    for dependent_group in dependent_groups:
        try:
            _check_group_reference(
                group=dependent_group,
                ref_entity=ref_entity,
                group_map=group_map,
            )
        except GroupReferenceError as error:
            errors.append(error)

    return errors


def _check_person(person, group_map):
    if person['official']['is_dismissed']:
        return []

    errors = []
    broken_groups_ids = set()
    ref_entity = '<Person: {0[id]}, {0[login]}>'.format(person)

    dependent_groups = [item['group'] for item in person['groups']]
    dependent_groups.append(person['department_group'])

    for dependent_group in dependent_groups:
        try:
            _check_group_reference(
                group=dependent_group,
                ref_entity=ref_entity,
                group_map=group_map,
            )
        except GroupReferenceError as error:
            errors.append(error)
            broken_groups_ids.add(dependent_group.get('id'))

    if settings.SKIP_BROKEN_STAFF_DATA:
        person['groups'] = [
            group for group in person['groups']
            if group['group']['id'] not in broken_groups_ids
        ]
        return

    return errors


def filter_valid_objects(checker, objects, group_map):
    errors = []
    valid_objects = []
    for obj in objects:
        new_errors = checker(obj, group_map)

        if not new_errors:
            valid_objects.append(obj)
        else:
            errors.extend(new_errors)

    return valid_objects, errors


def skip_broken_groups(groups):
    errors = []
    group_count = len(groups)
    prev_group_count = 0
    while group_count != prev_group_count:
        group_map = generate_group_map(groups)
        groups, new_errors = filter_valid_objects(_check_group, groups, group_map)
        errors.extend(new_errors)
        group_count, prev_group_count = len(groups), group_count

    return groups, errors


def _errors_to_str(errors):
    return ', '.join(map(str, errors))


class StaffStructureValidator(Schema):
    persons = f.List(f.Field())
    groups = f.List(f.Field())

    @decorators.post_dump()
    def check_staff_structure_consistency(self, data, **kwargs):
        groups = data['groups']
        persons = data['persons']

        broken_groups_errors = []
        if settings.SKIP_BROKEN_STAFF_DATA:
            groups, broken_groups_errors = skip_broken_groups(groups)

        group_map = generate_group_map(groups)

        valid_groups, groups_errors = filter_valid_objects(_check_group, groups, group_map)
        valid_persons, persons_errors = filter_valid_objects(_check_person, persons, group_map)

        all_groups_errors = broken_groups_errors + groups_errors

        if not settings.SKIP_BROKEN_STAFF_DATA and (all_groups_errors or persons_errors):
            raise ValidationError(_errors_to_str(all_groups_errors + persons_errors))

        data['groups'] = valid_groups
        data['persons'] = valid_persons
        data['groups_errors'] = all_groups_errors
        data['persons_errors'] = persons_errors

        return data


class PersonFetchValidator(Schema):
    class PersonOfficial(Schema):
        is_dismissed = f.Bool(required=True)
        is_robot = f.Bool(required=True)
        join_at = f.Str()

    class PersonName(Schema):
        class First(Schema):
            en = f.Str(required=True)

        class Last(Schema):
            en = f.Str(required=True)

        first = f.Nested(First)
        last = f.Nested(Last)

    class Environment(Schema):
        shell = f.Str()

    class DepartmentGroup(Schema):
        class Department(Schema):
            id = f.Int(allow_none=True)

        id = f.Int(required=True)
        department = f.Nested(Department)

    class GroupDict(Schema):
        class Group(Schema):
            id = f.Int(required=True)
            type = f.Str(required=True)
            is_deleted = f.Bool(required=True)

        group = f.Nested(Group)

    class Key(Schema):
        key = f.Str(required=True)

    id = f.Int(required=True)
    login = f.Str(required=True)
    official = f.Nested(PersonOfficial)
    name = f.Nested(PersonName, required=True)
    environment = f.Nested(Environment)
    department_group = f.Nested(DepartmentGroup)
    groups = f.Nested(GroupDict, many=True, allow_none=True, required=True)
    keys = f.Nested(Key, many=True, allow_none=True)

    @decorators.validates_schema(skip_on_field_errors=True)
    def validate_department_id(self, value, **kwargs):
        if (not value['official']['is_dismissed']
                and value['department_group']['department']['id'] is None):
            raise ValidationError("non-fired person has null department_group.group.id")
        return value

    def handle_error(self, error, data, **kwargs):
        error = error.messages
        error_keys = sorted(error.keys())
        error = [{'person': data[key], 'error': error[key]} for key in error_keys]

        exc = ValidationError(error)
        exc.valid_data = [data[i] for i in range(len(data)) if i not in error_keys]
        raise exc


class WikiGroupFetchValidator(Schema):
    id = f.Int(required=True)
    type = f.Str(validate=v.Regexp('^wiki$'))
    url = f.Str(required=True)
    department = f.Field()
    ancestors = f.Field()
    parent = f.Field()
    service = f.Field()


class ServiceGroupFetchValidator(Schema):
    class Service(Schema):
        id = f.Int(required=True)

    id = f.Int(required=True)
    type = f.Str(validate=v.Regexp('^service$'))
    url = f.Str(required=True)
    department = f.Field()
    ancestors = f.Field()
    parent = f.Field()
    service = f.Nested(Service)


class ServiceRoleGroupFetchValidator(Schema):
    class Parent(Schema):
        class Service(Schema):
            id = f.Int(required=True)
        id = f.Field()
        department = f.Field()
        service = f.Nested(Service)

    id = f.Int(required=True)
    type = f.Str(validate=v.Regexp('^servicerole$'))
    url = f.Str(required=True)
    department = f.Field()
    ancestors = f.Field()
    parent = f.Nested(Parent, required=True)
    service = f.Field()


class DepartmentGroupFetchValidator(Schema):
    class Department(Schema):
        id = f.Int(required=True)

    class Ancestor(Schema):
        class Department(Schema):
            id = f.Int(required=True)

        id = f.Int(required=True)
        is_deleted = f.Bool()
        department = f.Nested(Department)

    class Parent(Schema):
        class Department(Schema):
            id = f.Int(required=True)

        id = f.Int(required=True)
        department = f.Nested(Department)
        service = f.Field()

    id = f.Int(required=True)
    type = f.Str(validate=v.Regexp('^department$'))
    url = f.Str(required=True)
    department = f.Nested(Department, required=True)
    ancestors = f.Nested(Ancestor, many=True, allow_none=True)
    parent = f.Nested(Parent, allow_none=True)
    service = f.Field()


class GroupFetchValidator(CauthOneOfSchema):
    type_schemas = {
        'department': DepartmentGroupFetchValidator,
        'service': ServiceGroupFetchValidator,
        'servicerole': ServiceRoleGroupFetchValidator,
        'wiki': WikiGroupFetchValidator,
    }

    def get_data_type(self, data):
        return data['type']

    @decorators.validates_schema(skip_on_field_errors=True)
    def svc_group_has_service_id(self, value):
        if value['type'] not in ('svc', 'svcrole'):
            return value

        if isinstance(value['service_id'], int):
            return value

        raise ValidationError("Service group must have service_id")

    def handle_error(self, error, data):
        error = error.messages
        error_keys = error.keys()
        error_msg = [{'group': data[key], 'error': error[key]} for key in error_keys]

        exc = ValidationError(error_msg)
        exc.valid_data = [data[i] for i in range(len(data)) if i not in error_keys]
        raise exc


class PersonValidator(Schema):
    login = f.Str(required=True)
    gid = f.Int(validate=v.Range(min=20000))  # id группы для ldap
    uid = f.Int()  # id пользователя для ldap
    name = f.Str(allow_none=True)  # не импортируется
    first_name = f.Str()
    last_name = f.Str()
    shell = f.Str()
    is_fired = f.Bool(required=True)
    is_robot = f.Bool(required=True)
    pub_keys = f.List(f.Str())  # список публичных ssh-ключей
    join_date = f.Str()  # дата приёма на работу


class GroupValidator(Schema):
    name = f.Str(required=True)  # имя группы с префиксом (dpt_, wiki_)
    gid = f.Int(validate=v.Range(min=20000))  # id группы для ldap
    parent_gid = f.Int(validate=v.Range(min=20000), allow_none=True)  # id родительской группы
    type = f.Str(validate=v.Regexp('^(dpt|wiki|svc|svcrole)$'))  # один из четырех типов
    ancestors = f.List(f.Int(validate=v.Range(min=20000)))  # список id всех родительских групп
    members = f.List(f.Str())  # список логинов
    service_id = f.Int(allow_none=True)
    staff_id = f.Int()


# структура файла staff.json
class StaffImportValidator(Schema):
    persons = f.Nested(PersonValidator, many=True)  # список всех сотрудников
    groups = f.Nested(GroupValidator, many=True)  # список групп всех типов
