import subprocess
import logging
import tempfile
import datetime
import time
import json
import random

from ldap.modlist import modifyModlist
import ldif

from django.conf import settings
from django.utils.module_loading import import_string

from infra.cauth.server.master.utils.tasks import task
from infra.cauth.server.master.utils.mongo import get_mongo_database
from infra.cauth.server.master.importers import registry
from infra.cauth.server.master.importers.fetch import fetch_group
from infra.cauth.server.master.utils.dns_status import create_or_update_dns_status
from infra.cauth.server.master.api.models import Server
from infra.cauth.server.master.files.models import S3File
from infra.cauth.server.common.alchemy import Session


@task(dedicated_logger=False)
def update_watchdog():
    f = tempfile.TemporaryFile(mode='w')
    _ldif = ldif.LDIFWriter(f)

    new = f'wdog_{int(time.time())}'.encode()
    mod = modifyModlist(
        {'sn': {''}, 'cn': {''}},
        {'sn': {new}, 'cn': {new}},
    )
    _ldif.unparse('uid=cauth_watchdog,ou=people,dc=yandex,dc=net', mod)

    f.seek(0)

    ldapmodify = subprocess.Popen(
        args=['/usr/bin/ldapmodify', '-h', settings.LDAP_SERVER,
              '-D', settings.LDAP_USER, '-w', settings.LDAP_PASSWORD],
        stdin=f,
        stderr=subprocess.PIPE,
    )
    stdout, stderr = ldapmodify.communicate()

    if ldapmodify.returncode:
        raise RuntimeError(
            "ldapmodify exited with non-zero code {}: {}".format(
                ldapmodify.returncode, stderr))


def run_import_lock_name_getter(task_name, group_name, **kwargs):
    return ':'.join((task_name, group_name))


@task(use_lock=False, is_subtask=True, time_limit=120 * 60)
def run_import_subtask(cls_path, filename, suite_run_id):
    cls = import_string(cls_path)
    data = S3File.get_obj_data(filename)
    data = json.loads(data)

    try:
        instance = cls(data, suite_run_id=suite_run_id)
        instance.run()
    finally:
        if filename:
            S3File.objects.filter(name=filename).delete()


@task(lock_name=run_import_lock_name_getter)
def run_import(group_name, insane_mode=False):
    task_name = '.'.join((__name__, 'run_import'))
    logger = logging.getLogger(task_name)

    group = registry.SuiteGroup.registry[group_name]
    suites = [scls(logger=logger, insane_mode=insane_mode)
              for scls in list(group.suites.values())]

    data_filename = None
    acquired_suites = []
    try:
        for suite in suites:
            if suite.acquire_lock():
                acquired_suites.append(suite)

        if not acquired_suites:
            return 'all suites of import group "{}" are locked'.format(
                group_name)

        now = datetime.datetime.now().isoformat().replace(':', '.')
        data_filename = '_'.join((group.name, now))

        fetch_group(group.name, filename=data_filename, logger=logger)

        for suite in acquired_suites:
            try:
                suite.run(data_filename)
            except Exception:
                logger.exception("Import failed: {} -> {}".format(
                    group.name, suite.TARGET))
                raise

    finally:
        if data_filename:
            S3File.objects.filter(name=data_filename).delete()
        for suite in acquired_suites:
            suite.release_lock()


@task(dedicated_logger=False)
def cleanup_task_logs():
    db = get_mongo_database()
    db['task_logs'].remove({
        'started': {
            '$lt': datetime.datetime.now() - datetime.timedelta(days=30)
        },
    })


RAND_DAYS_INTERVAL = 7


def _random_expire_date():
    live_seconds = random.randrange(RAND_DAYS_INTERVAL * 24 * 60 * 60)
    return datetime.datetime.now() + datetime.timedelta(seconds=live_seconds)


@task(dedicated_logger=False, use_lock=False)
def update_dns_statuses(start_id, end_id):
    servers = Session.query(Server) \
        .filter(Server.id >= start_id) \
        .filter(Server.id <= end_id) \
        .order_by(Server.id) \
        .all()
    try:
        for server in servers:
            server_flags = create_or_update_dns_status(Session, server)
            server_flags.dns_status_expire_at = _random_expire_date()
    finally:
        Session.commit()
