from datetime import datetime, timedelta

import constance
from metrics_framework.decorators import metric
from sqlalchemy import or_, and_, not_

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import Server, ServerGroup
from infra.cauth.server.master.utils.mongo import get_mongo_database
from infra.cauth.server.master.settings import CAUTH_IDM_PUSH_IGNORE_DOMAINS


def is_in_ignorelist(server):
    return or_(*[
        server.fqdn.endswith(suffix) for suffix in CAUTH_IDM_PUSH_IGNORE_DOMAINS
    ])


def is_not_pushed_q(model):
    return or_(
        and_(model.last_push_ended_at.is_(None), model.first_pending_push_started_at.isnot(None)),
        and_(model.last_push_ended_at.isnot(None), model.last_push_ended_at < model.first_pending_push_started_at)
    )


def is_pushed_between_q(model, start_dt, end_dt):
    return and_(
        model.last_push_ended_at.isnot(None),
        model.last_push_ended_at > start_dt,
        model.last_push_ended_at <= end_dt,
        model.last_push_ended_at > model.first_pending_push_started_at,
    )


@metric('servers_without_tracked_pushes')
def count_servers_without_tracked_pushes():
    session = Session()
    servers_count = session.query(Server).filter(and_(
        not_(is_in_ignorelist(Server)),
        Server.first_pending_push_started_at.is_(None),
    )).count()
    return [{'slug': 'untracked_servers_count', 'value': servers_count}]


@metric('groups_without_tracked_pushes')
def count_groups_without_tracked_pushes():
    session = Session()
    groups_count = session.query(ServerGroup).filter(ServerGroup.first_pending_push_started_at.is_(None)).count()
    return [{'slug': 'untracked_groups_count', 'value': groups_count}]


@metric('not_pushed_hosts')
def count_not_pushed_hosts():
    session = Session()
    servers_count = session.query(Server).filter(is_not_pushed_q(Server)).count()
    groups_count = session.query(ServerGroup).filter(is_not_pushed_q(ServerGroup)).count()
    return [{'slug': 'not_pushed_count', 'value': servers_count + groups_count}]


@metric('hosts_count')
def count_hosts():
    session = Session()
    return [{'slug': 'hosts_count', 'value': session.query(Server).count() + session.query(ServerGroup).count()}]


@metric('queue_size')
def estimate_queue_size():
    return [{'slug': 'queue_size', 'value': get_mongo_database().messages.count()}]


@metric('push_to_idm_time')
def calculate_idm_push_time():
    end = datetime.now()
    start = end - timedelta(minutes=constance.config.PUSH_TO_IDM_TIME_METRIC_PERIOD_MINUTES)
    session = Session()
    servers_times = list(session.query(Server).filter(is_pushed_between_q(Server, start, end)))
    groups_times = list(session.query(ServerGroup).filter(is_pushed_between_q(ServerGroup, start, end)))
    times = sorted([(obj.last_push_ended_at - obj.first_pending_push_started_at).total_seconds() for obj in
                    servers_times + groups_times])
    last_index = len(times) - 1
    return [
        {'slug': 'push_time_100', 'value': times[last_index]},
        {'slug': 'push_time_99', 'value': times[int(last_index * 0.99)]},
        {'slug': 'push_time_90', 'value': times[int(last_index * 0.9)]}
    ]
