import datetime

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import ImportStatus, SkippedStaffInconsistencies

from infra.cauth.server.master.importers.registry import SuiteGroup
from infra.cauth.server.master.monitorings.views.base import BaseMonitoringView


class BaseImportView(BaseMonitoringView):
    NORMAL_LAG = datetime.timedelta(hours=1)
    SUITE_GROUP_NAME = None

    def monitoring(self):
        return self.import_monitoring(self.SUITE_GROUP_NAME)

    def import_monitoring(self, suite_group_name):
        suite_group = SuiteGroup.registry[suite_group_name]
        targets = list(suite_group.suites.keys())

        now = datetime.datetime.now()

        status_query = (
            Session.query(ImportStatus.target, ImportStatus.last_import)
            .filter(
                ImportStatus.suite == suite_group.name,
                ImportStatus.target.in_(targets)
            )
        )

        errors = []
        for target, last_import in status_query:
            if now - last_import > self.NORMAL_LAG:
                errors.append('last {suite}.{target} import finished on {last_import}'.format(
                    suite=suite_group_name,
                    target=target,
                    last_import=last_import,
                ))

        return errors


class StaffImportView(BaseImportView):
    SUITE_GROUP_NAME = 'staff'


class ServersImportView(BaseImportView):
    NORMAL_LAG = datetime.timedelta(hours=2)
    SUITE_GROUP_NAME = 'servers'


class SkippedInconsistenciesView(BaseMonitoringView):
    def monitoring(self):
        return [
            error
            for error, in (
                SkippedStaffInconsistencies.query
                .with_entities(SkippedStaffInconsistencies.error)
                .all()
            )
        ]
