import abc
import collections
import datetime

from django.conf import settings
from requests.exceptions import RequestException
from sqlalchemy import func, orm, or_

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import Access, User, Group, UserGroupRelation

from infra.cauth.server.master.notify.recipients import EmailRcpt, EmailCopy
from infra.cauth.server.master.notify.transports import EmailTransport
from infra.cauth.server.master.utils.idm import build_idm_form_url
from infra.cauth.server.master.utils.nda import shorten_url
from django.template.loader import get_template
from django.utils.translation import ugettext_lazy

NotificationRule = collections.namedtuple('NotificationRule', (
    'dst',
    'type',
    'sudo_role',
))


def qualify_email(login):
    return '%s@yandex-team.ru' % login


class Notification(object, metaclass=abc.ABCMeta):
    """Класс для аггрегации одинаковых по назначению event'ов в одно письмо или
    staff-тикет. Два разных NotifyEvent относятся к одному Notification, если
    Notification.get_key(event1) == Notification.get_key(event2)"""
    template = None
    subject = None

    def __init__(self, context=None, events=None, fallback=None, recipients=()):
        self.recipients = set()

        for recipient in recipients:
            self.recipients.add(recipient)

        self.context = context or {}
        self.events = events or []
        self.fallback = fallback

        self._body = None

    @classmethod
    def get_key(cls, event):
        raise NotImplementedError

    @abc.abstractmethod
    def _create(self, key):
        pass

    @abc.abstractmethod
    def _update(self, event):
        pass

    @classmethod
    def create(cls, key):
        obj = cls()
        obj.recipients.add(EmailCopy(settings.NOTIFICATIONS_GOLEM_CC))

        obj._create(key)

        return obj

    def update(self, event):
        self.events.append(event)

        self._update(event)

    def compile(self):
        pass

    @property
    def body(self):
        if self._body is not None:
            return self._body

        template = get_template(self.template)
        self.context['events'] = self.events
        self._body = template.render(self.context)
        return self._body

    @property
    def is_ready(self):
        threshold = datetime.datetime.now() - datetime.timedelta(minutes=10)
        return any(event.timestamp < threshold for event in self.events)

    def ready_to_deliver(self):
        return True

    def deliver(self):
        self.compile()

        if self.ready_to_deliver():
            transport = EmailTransport()
            transport.deliver(self)


class RemovedFromGroupNotification(Notification):
    template = 'notify/removed_from_group.txt'
    subject = ugettext_lazy('CAuth: некоторые доступы будут отозваны')

    @classmethod
    def get_key(cls, event):
        return event.who

    @classmethod
    def _same_rules_count(cls, rule, user, user_gids):
        """Возвращает количество доступов аналогичных rule для пользователя
        user. Они должны истекать после настоящего удаления из группы"""
        today = datetime.date.today()
        until = today + datetime.timedelta(days=settings.HOLD_MEMBERSHIP_PERIOD)

        src_filter = (Access.src_user == user)
        if user_gids:
            src_filter = or_(
                Access.src_user == user,
                Access.src_group_id.in_(user_gids),
            )
        query = (
            Session.query(func.count(Access.id))
            .filter(
                Access.dst == rule.dst,
                Access.type == rule.type,
                Access.sudo_role == rule.sudo_role,
                src_filter,
                or_(
                    Access.until.is_(None),
                    Access.until >= until,
                )
            )
        )
        return Access.add_active_filter(query).scalar()

    @classmethod
    def _get_rules_for_removing(cls, user, group_names):
        """Возвращает доступы групп, из которых недавно вышел пользователь. Если
         пользователь вступил обратно в одну из групп то в учет её не берем."""
        return (
            Access.get_active_query()
            .join(Access.src_group)
            .join(UserGroupRelation)
            .filter(
                Group.name.in_(group_names),
                UserGroupRelation.user == user,
                UserGroupRelation.until.isnot(None),
            )
            .options(
                orm.contains_eager(Access.src_group),
                orm.joinedload(Access.sudo_role),
            )
        )

    def compile(self):
        user = User.query.filter(User.login == self.context['who']).first()

        memberships = (
            UserGroupRelation.query
            .filter(
                UserGroupRelation.user == user,
                UserGroupRelation.until.is_(None),
            )
        )
        user_gids = {membership.gid for membership in memberships}

        all_rules = []
        group_names = {event.what for event in self.events}
        for rule in self._get_rules_for_removing(user, group_names):
            if not self._same_rules_count(rule, user, user_gids):
                notify_rule = NotificationRule(
                    type=rule.type,
                    dst=rule.dst,
                    sudo_role=rule.sudo_role.spec if rule.type == 'sudo' else ''
                )
                self.context['groups'][rule.src_group.name].append(notify_rule)
                all_rules.append(rule)

        self.context['groups'] = list(self.context['groups'].items())
        form_url = build_idm_form_url(all_rules)
        try:
            form_url = shorten_url(form_url)
        except RequestException:
            pass

        self.context['form_url'] = form_url

    @property
    def is_ready(self):
        interval = datetime.timedelta(hours=settings.CAUTH_NOTIFICATIONS_REMOVAL_GROUP_INTERVAL)
        threshold = datetime.datetime.now() - interval
        return all(event.timestamp < threshold for event in self.events)

    def ready_to_deliver(self):
        return bool(self.context['groups'])

    def _create(self, key):
        who = key

        delta = datetime.timedelta(days=settings.HOLD_MEMBERSHIP_PERIOD)
        self.context['expire_date'] = str(datetime.date.today() + delta)
        self.context['who'] = who
        self.context['groups'] = collections.defaultdict(list)
        self.context['form_url'] = ''

        email = qualify_email(who)
        self.recipients.add(EmailRcpt(email))

    def _update(self, event):
        pass


event_notification = {
    'removed_from_group': RemovedFromGroupNotification,
}
