import re

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.models import Group, UserGroupRelation

svc_admin_pattern = re.compile('^svc_.+_administration$')
openstack_source_pattern = re.compile('^os-')


class Recipient(object):
    @property
    def _unique_field(self):
        raise NotImplementedError

    def _as_tuple(self):
        return type(self).__name__, getattr(self, self._unique_field)

    def __repr__(self):
        return '{}({})'.format(*self._as_tuple())

    def __hash__(self):
        return hash(self._as_tuple())

    def __eq__(self, other):
        if isinstance(other, Recipient):
            return self._as_tuple() == other._as_tuple()
        else:
            return False


class EmailRcpt(Recipient):
    _unique_field = 'email'

    def __init__(self, email):
        self.email = email


class EmailCopy(EmailRcpt):
    pass


def only_admins(users):
    if not users:
        return []

    uids = [u.uid for u in users]
    names_query = (
        Session.query(UserGroupRelation.uid, Group.name)
        .join(Group)
        .filter(UserGroupRelation.uid.in_(uids))
    )

    admin_uids = set()
    for uid, group_name in names_query:
        if svc_admin_pattern.match(group_name):
            admin_uids.add(uid)

    return [user for user in users if user.uid in admin_uids]
