import logging
import smtplib
from email.charset import Charset
from email.encoders import encode_base64
from email.header import Header
from email.mime.text import MIMEText

from django.conf import settings
from django.utils.encoding import force_text

from .recipients import EmailRcpt, EmailCopy

logger = logging.getLogger(__name__)


class TransportError(Exception):
    pass


class DeliveryImpossibleError(TransportError):
    def __init__(self, message, response=None):
        self.message = message
        self.response = response


class TransportBase(object):
    def deliver(self, notification):
        raise NotImplementedError


class EmailTransport(TransportBase):
    def __init__(self):
        self._smtp = None

    @property
    def smtp(self):
        if self._smtp is None:
            self._smtp = smtplib.SMTP('localhost')
        return self._smtp

    def deliver(self, notification):
        message = self._make_message(notification)

        all_emails = {r.email for r in notification.recipients
                      if isinstance(r, EmailRcpt)}

        # Вилка для тестинга (отправляет письма только на один адрес)
        if settings.NOTIFICATIONS_FORCE_RCPT:
            all_emails = {settings.NOTIFICATIONS_FORCE_RCPT}

        # На девеле письма должны только логироваться
        if settings.NOTIFICATIONS_LOG_ONLY:
            logger.info(message)
            return

        self.smtp.sendmail(settings.NOTIFICATIONS_SENDER, all_emails, message)

    def _make_message(self, notification):
        charset = Charset('utf-8')
        message = MIMEText(
            _text=None,
            _charset='utf-8',
        )

        message.set_payload(payload=notification.body.encode('utf-8'))
        encoded_subject = notification.subject.encode('utf-8')

        if settings.NOTIFICATIONS_ENCODE:
            del message['Content-Transfer-Encoding']
            encode_base64(message)
            subject_header = Header(encoded_subject, charset)
        else:
            message['Content-Transfer-Encoding'] = '8bit'
            subject_header = encoded_subject

        rcpt_emails = []
        copy_emails = []
        for rcpt in notification.recipients:
            if isinstance(rcpt, EmailCopy):
                copy_emails.append(rcpt.email)
            elif isinstance(rcpt, EmailRcpt):
                rcpt_emails.append(rcpt.email)

        message['From'] = settings.NOTIFICATIONS_SENDER
        message['To'] = ", ".join(rcpt_emails)
        message['Cc'] = ", ".join(copy_emails)
        message['Subject'] = force_text(subject_header)

        return force_text(message.as_bytes())
