import pickle
import os
import atexit
import shutil

from django.core.cache.backends.base import BaseCache, DEFAULT_TIMEOUT
from django.core.cache.backends.filebased import FileBasedCache


class PickledCacheBackend(BaseCache):
    def __init__(self, host, *args, **kwargs):
        BaseCache.__init__(self, *args, **kwargs)

        self._modified = False
        self._filename = host
        if os.path.exists(self._filename):
            self._data = pickle.load(open(self._filename))
        else:
            self._data = {}

        atexit.register(self._save)

    def add(self, key, value, timeout=DEFAULT_TIMEOUT, version=None):
        if key in self._data:
            return False
        else:
            self._data[key] = value
            self._modified = True
            return True

    def get(self, key, default=None, version=None):
        return self._data.get(key)

    def set(self, key, value, timeout=DEFAULT_TIMEOUT, version=None):
        self._data[key] = value
        self._modified = True
        return True

    def delete(self, key, version=None):
        if key in self._data:
            self._modified = True
            del self._data[key]

    def get_many(self, keys, version=None):
        return {key: self._data[key] for key in keys if key in self._data}

    def has_key(self, key, version=None):
        return key in self._data

    def clear(self):
        self._modified = True
        self._data = {}

    def _save(self):
        if not self._modified:
            return

        swap_filename = self._filename + '.swap'
        pickle.dump(self._data, open(swap_filename, 'w'),
                    pickle.HIGHEST_PROTOCOL)
        shutil.move(swap_filename, self._filename)


class UnlimitedFileBasedCache(FileBasedCache):
    def _cull(self):
        pass
