import collections

from sqlalchemy.exc import IntegrityError

CreateRequest = collections.namedtuple('CreateRequest', ('model', 'attr', 'values'))
FetchResult = collections.namedtuple('FetchResult', ('request', 'value', 'obj'))
CreateResult = collections.namedtuple('CreateResult', ('obj', 'is_created'))


def _fetch(session, requests, for_update=False):
    # во избежание deadlock'ов следует брать блокировки в определённом порядке
    requests = list(requests)
    requests.sort(key=lambda req: (req.model.__name__, req.attr))

    results = []
    for request in requests:
        assert isinstance(request, CreateRequest)
        field = getattr(request.model, request.attr)
        query = session.query(request.model).filter(field.in_(request.values))
        if for_update:
            query = query.with_for_update()

        unused_values = set(request.values)
        for obj in query:
            attr_value = getattr(obj, request.attr)
            results.append(FetchResult(
                request=request,
                value=attr_value,
                obj=obj,
            ))
            unused_values.remove(attr_value)

        for value in unused_values:
            results.append(FetchResult(
                request=request,
                value=value,
                obj=None,
            ))

    return results


def get_or_create(session, model, attr_name, attr_value, for_update=False):
    request = CreateRequest(model, attr_name, [attr_value])
    return get_or_create_many(session, [request], for_update)[0]


def get_or_create_many(session, requests, for_update=False):
    """
    Выбирает или создаёт список объектов из базы.
    Обязательно делает это в одной сессии, да так,
    чтобы по завершении все созданные строки остались залоченными.
    Если выставить флаг for_update, то на существующие строки
    тоже будет взят лок.
    """

    def _inner_get_or_create():
        fetch_results = _fetch(session, requests, for_update)
        create_results = []
        new_instances = []
        for result in fetch_results:
            assert isinstance(result, FetchResult)
            if result.obj:
                create_results.append(CreateResult(result.obj, False))
                continue

            instance = result.request.model(**{result.request.attr: result.value})
            new_instances.append(instance)
            create_results.append(CreateResult(instance, True))

        if new_instances:
            session.add_all(new_instances)
            session.flush()

        return create_results

    while True:
        try:
            return _inner_get_or_create()
        except IntegrityError:
            session.rollback()
