import time
import requests
import logging
import hashlib

from django.conf import settings
from django.core.cache import caches


default_logger = logging.getLogger(__name__)


class HttpClient(object):
    _session = requests.Session()

    @classmethod
    def fetch(cls, url, oauth=False, logger=None, **kwargs):
        if oauth:
            kwargs['headers'] = kwargs.get('headers', {})
            kwargs['headers'].update({
                'Authorization': 'OAuth %s' % settings.OAUTH_TOKEN,
            })

        logger = logger or default_logger

        if 'timeout' not in kwargs:
            kwargs['timeout'] = settings.IMPORTERS_FETCH_TIMEOUT

        cache_backend = caches['http_client']
        cache_key = hashlib.sha1(url.encode('utf-8')).hexdigest()

        data = cache_backend.get(cache_key)
        if data is not None:
            return data

        retries_left = settings.IMPORTERS_FETCH_RETRIES

        # TODO(lavrukov) Это можно успешно сделать стандартными ретраями из requests
        while retries_left:
            try:
                logger.info("Fetching %s", url)
                response = cls._session.get(
                    url,
                    allow_redirects=False,
                    verify=settings.ROOT_CA_BUNDLE,
                    **kwargs
                )

                if response.status_code != 200:
                    raise requests.HTTPError(
                        "Invalid status code: %s" % response.status_code,
                        response=response,
                    )

                break
            except Exception:
                retries_left -= 1

                if retries_left:
                    msg = ("Failed to fetch %(url)s, "
                           "will retry in %(timeout)ss")
                    params = dict(
                        url=url,
                        timeout=settings.IMPORTERS_FETCH_RETRY_INTERVAL,
                    )
                    logger.exception(msg, params)

                    time.sleep(settings.IMPORTERS_FETCH_RETRY_INTERVAL)
                    continue
                else:
                    logger.exception("Failed to fetch %s", url)
                    raise

        cache_backend.set(cache_key, response.content)
        return response.content
