import datetime
import collections
import time
import uuid
import logging

from django.conf import settings

from infra.cauth.server.master.utils.mongo import get_mongo_database


Task = collections.namedtuple('Task', 'task args kwargs time_limit')

logger = logging.getLogger(__name__)


class SubtaskError(Exception):
    def __init__(self, tasks):
        self.tasks = tasks


class SubtaskPool(object):
    @property
    def collection(self):
        return get_mongo_database()['subtask_pool']

    def __init__(self, id=None, suite_run_id=None):
        self.id = id or str(uuid.uuid4())
        if not isinstance(suite_run_id, str):
            raise TypeError("SubtaskPool must be instantiated with suite_run_id")
        self.suite_run_id = suite_run_id
        self.tasks = []

    def apply(self, task, args=None, kwargs=None, time_limit=None):
        args = args or []
        kwargs = kwargs or {}
        kwargs['pool_id'] = self.id
        kwargs['suite_run_id'] = self.suite_run_id
        self.tasks.append(Task(task, args, kwargs, time_limit))

    def get_lock(self):
        from .tasks import locked_context, fake_locked_context
        if settings.DISABLE_TASK_LOCKING:
            return fake_locked_context()
        else:
            return locked_context(":".join((type(self).__name__, self.id)))

    def join(self):
        collection = self.collection

        # cleanup
        collection.remove({'pool_id': self.id})
        collection.remove({'created_at': {
            '$lt': datetime.datetime.now() - datetime.timedelta(days=10)}
        })

        with self.get_lock() as is_locked:
            if not is_locked:
                raise RuntimeError("Strangely could not acquire "
                                   "lock for subtask pool {}.".format(self.id))

            for task in self.tasks:
                result = task.task.apply_async(task.args, task.kwargs)
                self._register_subtask(task.task, result, task.time_limit)
                logger.info("Subtask {} added to queue".format(result.id))

            while True:
                unfinished_tasks = list(collection.find({
                    'pool_id': self.id,
                    'finished_at': None
                }))

                if not unfinished_tasks:
                    break

                now = datetime.datetime.now()
                for task in unfinished_tasks:
                    if not task['time_limit']:
                        continue

                    delta_timeout = datetime.timedelta(seconds=task['time_limit'])
                    if (task['started_at']
                            and now - task['started_at'] > delta_timeout):
                        self.report_error(
                            task_id=task['task_id'],
                            error='Timeout {} seconds'.format(task['time_limit']),
                        )

                time.sleep(1)

        failed_tasks = list(collection.find({
            'pool_id': self.id,
            'is_successful': False,
        }))

        collection.remove({'pool_id': self.id})

        if failed_tasks:
            raise SubtaskError(failed_tasks)

    def report_started(self, task_id):
        self._update_subtask(task_id, {
            'started_at': datetime.datetime.now(),
        })

    def report_finished(self, task_id):
        self._update_subtask(task_id, {
            'finished_at': datetime.datetime.now(),
            'is_successful': True,
        })

    def report_error(self, task_id, error):
        self._update_subtask(task_id, {
            'finished_at': datetime.datetime.now(),
            'is_successful': False,
            'error': error,
        })

    def _register_subtask(self, task, result, time_limit=None):

        self.collection.insert({
            'pool_id': self.id,
            'task_id': result.id,
            'task_name': task.name,
            'time_limit': time_limit or task.time_limit,
            'started_at': None,
            'finished_at': None,
            'is_successful': None,
            'error': None,
        })

    def _update_subtask(self, task_id, fields):
        query = {'pool_id': self.id, 'task_id': task_id}
        self.collection.update(query, {'$set': fields})


class FakeSubtaskPool(object):

    def __init__(self, id=None, suite_run_id=None):
        self.id = id or '1'
        self.suite_run_id = suite_run_id
        self.tasks = []

    def apply(self, task, args=None, kwargs=None, time_limit=None):
        args = args or []
        kwargs = kwargs or {}
        kwargs['suite_run_id'] = self.suite_run_id
        kwargs['pool_id'] = self.id
        self.tasks.append((task, args, kwargs))

    def get_lock(self):
        pass

    def join(self):
        for task, args, kwargs in self.tasks:
            task(*args, **kwargs)

    def report_started(self, task_id):
        pass

    def report_finished(self, task_id):
        pass

    def report_error(self, task_id, error):
        pass
