import abc
import hashlib
import netaddr
import re
import socket
import subprocess
from distutils.version import StrictVersion

from django.http import HttpResponse, HttpResponseBadRequest, HttpResponseNotFound, Http404
from django.template.loader import get_template
from django.utils.functional import cached_property
from django.utils.encoding import force_text
from django.views.generic import View

from infra.cauth.server.common.alchemy import Session
from infra.cauth.server.common.constants import FLOW_TYPE
from infra.cauth.server.common.models import (
    Access,
    Role,
    Server,
    Source,
)
from infra.cauth.server.public.api.helpers import get_access_dst_filters
from infra.cauth.server.public.constants import SOURCE_NAME


class RequestError(RuntimeError):
    def __init__(self, message):
        self.message = message


class BaseView(View, metaclass=abc.ABCMeta):
    template_name = None
    real_fqdn = force_text(subprocess.check_output([b'hostname', b'-f']).rstrip(b'\n'))

    fallback_fqdn_patterns = {
        r'\1.yandex.ru': re.compile(r'^(.*)\.yabs\.yandex\.ru$', re.IGNORECASE),
    }

    with_response_wrapper = True
    content_type = 'text/plain; charset=utf-8'

    @abc.abstractmethod
    def get_context_data(self):
        pass

    @cached_property
    def q_param(self):
        return self.request.GET.get('q') or None

    @cached_property
    def effective_remote_addr(self):
        if self.q_param is not None:
            try:
                return self.convert_mapped_ipv6(netaddr.IPAddress(self.q_param))
            except netaddr.AddrFormatError:
                pass

        return self.remote_addr

    @cached_property
    def effective_remote_fqdn(self):
        if self.q_param is not None:
            return self.q_param

        return self.remote_fqdn

    @cached_property
    def remote_fqdn(self):
        return socket.getfqdn(str(self.remote_addr))

    @staticmethod
    def convert_mapped_ipv6(ip):
        if (ip.is_ipv4_compat() or ip.is_ipv4_mapped())\
                and not ip.is_loopback():
            return ip.ipv4()

        return ip

    @cached_property
    def remote_addr(self):
        x_forwarded_for = self.request.META.get('HTTP_X_FORWARDED_FOR')
        value = (
            x_forwarded_for.split(',')[0].strip()
            if x_forwarded_for
            else
            self.request.META['REMOTE_ADDR']
        )
        return self.convert_mapped_ipv6(netaddr.IPAddress(value))

    @cached_property
    def client_checksum(self):
        return self.request.GET.get('md5')

    @cached_property
    def remote_server(self):
        fqdn = self.effective_remote_fqdn
        server = Server.query.filter_by(fqdn=fqdn).first()

        if server is not None:
            return server

        for replacement, pattern in self.fallback_fqdn_patterns.items():
            new_fqdn = pattern.sub(replacement, fqdn)
            server = Server.query.filter_by(fqdn=new_fqdn).first()

            if server is not None:
                return server

        return None

    def render_body(self):
        template = get_template(self.template_name).template
        context = self.get_context_data()
        return template.render(context).strip()

    def get(self, request, **kwargs):
        try:
            body = self.render_body()
        except RequestError as e:
            return HttpResponseBadRequest(
                force_text(e),
                content_type='text/plain; charset=utf-8',
            )
        except Http404 as e:
            return HttpResponseNotFound(
                force_text(e),
                content_type='text/plain; charset=utf-8',
            )
        if self.with_response_wrapper:
            checksum = hashlib.md5((body + '\n').encode('utf-8')).hexdigest()

            if checksum == self.client_checksum:
                return HttpResponse('OK', content_type='text/plain')

            template = get_template('api/response.txt').template
            context = {
                'body': body,
                'checksum': checksum,
                'real': self.real_fqdn,
            }
            body = template.render(context)

        return HttpResponse(
            body,
            content_type=self.content_type,
        )

    def error(self, message):
        raise RequestError(message)


class BaseSourceView(BaseView, metaclass=abc.ABCMeta):
    """ Костырь по CAUTH-944
    SOX системы теперь должны ходить в API с параметром sources, в котором должны быть указаны
    имена источников через запятую.
    В соответствии с переданными источниками фильтруются ответсвенные на хосты и доступы.
    """

    new_logic_version = StrictVersion('1.3.2')
    client_version_pattern = re.compile(r'^CAUTH/yandex-cauth-(?P<version>.+)$')

    @cached_property
    def client_version(self):
        user_agent = self.request.META.get('HTTP_USER_AGENT', '')
        if not user_agent.startswith('CAUTH/'):
            return None

        return user_agent

    @cached_property
    def client_version_is_old(self):
        if self.client_version is None:
            return True

        match = self.client_version_pattern.match(self.client_version)
        if not match:
            return True

        try:
            client_version = StrictVersion(match.group('version'))
        except ValueError:
            return True

        return client_version < self.new_logic_version

    @cached_property
    def sources_names(self):
        sources_names = self.request.GET.get('sources')
        if sources_names is None:
            return None

        # Для версий < 1.3.2 если передан 'sources=' то считаем что у хоста дефолтная логика
        # Для версий >= 1.3.2 считаем что хост не доверяет никому
        if not sources_names:
            return None if self.client_version_is_old else set()

        sources_names = set(source_name.strip() for source_name in sources_names.split(','))

        # default - технический источник, игнорируем
        sources_names -= {SOURCE_NAME.DEFAULT}

        if self.client_version_is_old:
            # Для версий < 1.3.2 если передан 'sources=idm' то считаем что у хоста дефолтная логика
            if sources_names == {SOURCE_NAME.IDM}:
                return None
            # Для версий < 1.3.2 если переданы sources то мы считаем, что хост доверяет idm
            sources_names.add(SOURCE_NAME.IDM)

        return sources_names

    @cached_property
    def sources(self):
        if self.remote_server is not None and self.remote_server.flow != FLOW_TYPE.CLASSIC:
            return self.remote_server.trusted_sources

        if self.sources_names is None:
            return None
        if not self.sources_names:
            return []

        sources = list(Session.query(Source).filter(Source.name.in_(self.sources_names)))

        if len(sources) != len(self.sources_names):
            invalid_sources = ', '.join(self.sources_names - {source.name for source in sources})
            self.error('Sources {} does not exists'.format(invalid_sources))

        return sources

    @property
    def can_use_access(self):
        if self.remote_server and self.remote_server.flow == FLOW_TYPE.BACKEND_SOURCES:
            backend_sources_names = {s.name for s in self.remote_server.trusted_sources}
            return bool(SOURCE_NAME.IDM_SOURCES & backend_sources_names)

        return self.sources_names is None or bool(SOURCE_NAME.IDM_SOURCES & self.sources_names)

    @property
    def can_make_context(self):
        return True

    @abc.abstractproperty
    def default_context(self):
        return {}

    @abc.abstractmethod
    def make_context(self):
        return {}

    @staticmethod
    def make_rule(name, spec, is_user):
        prefix = '' if is_user else '%'
        return '{}{} {}'.format(prefix, name, spec)

    def get_sudoers(self):
        query = (
            Session.query(Access.src, Role.spec, Access.src_user_id.isnot(None))
            .join(Access.sudo_role)
            .filter(*get_access_dst_filters(self.remote_server, 'sudo', self.sources))
        )
        rules = {self.make_rule(*values) for values in query}
        # Правила с NOPASSWD отдаем в конце списка CAUTH-1358
        return (sorted({rule for rule in rules if 'NOPASSWD:' not in rule})
                +
                sorted({rule for rule in rules if 'NOPASSWD:' in rule}))

    def get_context_data(self):
        if self.remote_server is None:
            raise Http404('Host not found')

        context = self.default_context
        if self.can_make_context:
            context = self.make_context()

        result = {
            'fqdn': self.effective_remote_fqdn,
        }
        result.update(context)

        return result
